/* printer_get_dev_file.c
   This method provides the default printer device on the system. At this point the first device found will be returned, 
   having no concept of a default device (yet).
   change history
   05/03/2025 initial version
   12/12/2025 updated to read a default printer device from /etc/printcap to support cases where multiple PIO ports exist
   12/16/2025 updated to read standard printcap file
   12/17/2025 updated to handle multiple line entries
*/

#include <printer_private.h>
#include <dylib.h>
#include <cache_private.h>

int printer_get_dev_file (const char *printer, char *path) {
 
   const char line_sep[4] = {':', 92, 0, 0};
   char printers[80];
   int r = 1;

   FILE *f = dylib.fopen (PRINTCAP_FILE, "r");
   if (f) {
      char u[128];
      char s[128];

      dylib.strcpy (s, "");                                         // initialize s
      while (r && dylib.fgets (u, sizeof (u), f)) {                 // loop until the printer was found or no new lines read
         u[dylib.strcspn (u, "\r\n")] = 0x00;                       // kill off the cr/lf characters
         char *p = strstr (u, line_sep);                            // find a line seperator, if it exists
         if (p) {                                                   // test p
            *p = 0x00;                                              // kill off the line separator 
         }
         dylib.strcat (s, u);                                       // append u to s so the printer data can be processed in a single string
         if (!p) {                                                  // if no line seperator was found, it's time to process the line
            if (s[0] != '#') {                                      // if not a comment then process the line
               char *e = strtok (s, ":");                           // get the first token set
               dylib.strcpy (printers, e);                          // capture the list of printers for this entry
               while (e) {                                          // loop through the options
                  if (strstr (e, "lp=")) {                          // search for the lp option
                     char *q = e + 3;                               // skip over the lp= text
                     dylib.strcpy (path, q);                        // capture the path
                     break;                                         // exit the loop since this was found
                  }
                  e = strtok (NULL, ":");                           // get the next token
               }
               e = strtok (printers, "|");                          // get the first printer name for this line
               while (r && e) {                                     // loop until the printer is found or there's no more printers
                  r = dylib.strcmp (printer, e);                    // assess this name
                  e = strtok (NULL, "|");                           // get the next token
               }
            }
            dylib.strcpy (s, "");
         }
      }
      dylib.fclose (f);                                             // close the file
   }

   return r;                                                     // return the result
}
