/* rmdir.c
   This method removes a directory per the unix standard.
   change log:
   05/24/2025 initial version
   05/29/2025 updated return value to not conflict with the DSRL2's operation not permitted value
*/ 

#include <dsrl2_private.h>
#include <sys/stat.h>
#include <dirent.h>
#include <libgen.h>
#include <string.h>
#include <conversion.h>
#include <dylib.h>

int rmdir (const char *path, unsigned int mode) {

   int r = 1; // start with the assumption the path doesn't exist
              // 2 = dir exists but isn't empty
              // 0 = success

   // Note: it is necessary to ensure that the parent exists as there is a bug in the DSR that a subsequent set_directory will succeeed even when the path doesn't exist.
   // Example: set_directory WDS1.home.marko.       (exists)
   //          rmdir         abc
   //          set_directory WDS1.home.marko.xyz.   (doesn't exist)
   //          rmdir         123

   DIR *d;
   struct dirent *e;

   bool found  = false;
   bool is_dir = false;

   // ensure parent directory exists
   d = opendir (dirname ((char *) path));
   if (d) {
      // determine if the path exists
      while ((e = readdir (d))) {
         if (!dylib.strcmp (basename ((char *) path), e->d_name)) {
            found = true;
            // determine if this is a directory
            is_dir = (int) ftol (e->d_type) == 6;
            break;
         }
      }
      closedir (d);

      // process if the entry was found
      if (found) {
         // make sure it's a directory
         if (is_dir) {
            // ensure the directory to be deleted is empty
            d = opendir (path);
            if (d) {
               // ensure the directory is empty
               if (readdir (d)) {
                  r = 2;  // there are contents remaining. can't delete this directory
               }
               closedir (d);
               if (r != 2) { // if empty attempt deletion
                  r = dsrl2_rmdir (path);
               }
            }
         } else {
            r = 4;
         }
      }
   }

   return r;
}
