/* rt.c
   This file implements [near] real-time support methods.
   06/23/2023 initial version
*/

#include <vdp.h>
#include <speech.h>
#include <stdbool.h>
#include <constants.h>
#include <sequencer_private.h>
#include <rt.h>

#define RT99_METHODS_PER_CYCLE 4

typedef struct {
   rt99_method_t *method[RT99_METHODS_PER_CYCLE];
   int count;
} rt99_methods_t;

typedef struct {
   rt99_methods_t methods[RT99_CYCLES];
   long tick;
   bool cont;
} rt99_ctl_t;

rt99_ctl_t rt99_ctl;

void rt99_ctl_init () {
   int i;
   int j;
   for (i = 0; i < RT99_CYCLES; i++) {
      for (j = 0; j < RT99_METHODS_PER_CYCLE; j++) {
         rt99_ctl.methods[i].method[j] = 0x0000;
      }
      rt99_ctl.methods[i].count = 0;
   }
   rt99_ctl.tick = 0;
   rt99_ctl.cont = true;
}

void rt99_add_method (rt99_method_t *method, int minor_cycle) {
   rt99_ctl.methods[minor_cycle].method[rt99_ctl.methods[minor_cycle].count] = method;
   rt99_ctl.methods[minor_cycle].count++;
}

void rt99_update_tick () {
   rt99_ctl.tick++;
}

void rt99_init () {
   rt99_ctl_init ();
   VDP_INT_HOOK = &rt99_update_tick; // add the isr handler
}

void rt99_sched_exit () {
   rt99_ctl.cont = false;
}

void rt99_sched_run () {

   long rt_tick_last = 0;
   int cycle_minor   = 0;

   long rt_tick_capture;
   long cycle_diff;

   int i;

   while (rt99_ctl.cont) {
      rt_tick_capture = rt99_ctl.tick;
      cycle_diff      = rt_tick_capture - rt_tick_last;

      while (cycle_diff) {
         for (i = 0; i < rt99_ctl.methods[cycle_minor].count; i++) {
            rt99_ctl.methods[cycle_minor].method[i] ();
         }
         cycle_minor++;
         if (cycle_minor >= RT99_CYCLES) cycle_minor = 0;
         cycle_diff--;
      }
      rt_tick_last = rt_tick_capture;
      sequencer_update ();
   }
}
