/* shmmgr.c
   This set of library methods demonstrates shared memory usage in a practical way.

   This library allows for complete use of shared memory, which is limited to three shared memory segments, each of which are up to
   192 KB each. Thus the total supported is 576K. The allocations are continued in shared memory segments, which are further sub-
   dividied into pages. The algorithm for reading elements caches the current segment and page to minimize costly swaps.

   Initialization is performed by shmmgr_init ().

   Individual allocations are performed using shmmgr_alloc (). It first finding a free element in the alloc matrix, where the bit is
   set to 0 (unused). The search continues until one is found. Shared memory segments are allocated as needed. The resultant 
   allocation returns only an index in the cumulative array. The search begins at the last known position where a free item was 
   found to exist.

   To access the allocation, shmmgr_map_and_return_ptr () is used to return a pointer. This pointer is guaranteed to be stable only
   until the next call to shmmgr_map_and_return_ptr (), which may swap the shared memory segment and/or page. If interaction between
   multiple allocations are necessary, it is recommended that copies of each be made, and the results copied back.

   Finally the memory allocations can be ended with shmmgr_term (), resulting in deallocation of all allocated shared memory 
   segments.
 
   The size of the data to be managed must be provided. The library internally resizes it a power of 2 to ensure that library
   calculations for location, etc. work mathmatically.

   limitations

   - memory allocation is fairly course, allocating 192 KB at a time

   change history
   10/09/2025 initial version
   10/10/2025 added retention of the first known free search position in the allocation index as a performance improvement
              documentation and format updates
   10/21/2025 removed calc_power
   10/23/2025 added dylib reference
   10/25/2025 shift alternatives to mpy, div and mod
*/

#include <sys/ipc.h>
#include <sys/shm.h>
#include <stdlib.h>
#include <dylib.h>
#include <util.h>

// helpful declarations used by these library methods

#define SHMMGR_REGION_ALLOC_SIZE 196608      // each region is 192 KB long
#define SHMMGR_UNALLOCATED       -1          // constant for a shm key not yet allocated
#define SHMMGR_ALLOC_FULL        0xff        // constant for a byte that indicates the sequence of 8 fields rep'd as bits are alloc'd

// initializes the shared memory tracking structures
int shmmgr_init (shmmgr_t *shmmgr, const char *name, int value_type_size) {

   int r = 0;

   memset (shmmgr, 0x00, sizeof (shmmgr_t));               // initialize the storage
                                                           // note: alloc_matrix and alloc_matrix_first_free are set to zero 

   if (!r) {                                               // capture the name, error out if spec not met
      if (dylib.strlen (name) <= 4) {
         strcpy (shmmgr->name, name);
      } else {
         r = 1;
      }
   }

   if (!r) {                                               // pre-calculate all the necessary info ahead of actual allocations
      for (int i = 0; i < SHMMGR_REGION_MAX; i++) {        // set the keys to unallocated
         shmmgr->key[i] = SHMMGR_UNALLOCATED;
      }

      shmmgr->alloc_size = calc_power (value_type_size);   // alloc size, rounded up to nearest power

      if (shmmgr->alloc_size <= SHMLBA) {
         shmmgr->region_div = SHMMGR_REGION_ALLOC_SIZE /   // calculate the region divisor
                           shmmgr->alloc_size;
         shmmgr->page_div   = SHMLBA / shmmgr->alloc_size; // calculate the page divisor
         shmmgr->alloc_max  = shmmgr->region_div * 
                           SHMMGR_REGION_MAX;
         shmmgr->cur_region = -1;                          // set the current cached region and page to unused (really out of range)
         shmmgr->cur_page   = -1;
      } else { 
         r = 2;
      }
   }

   return r;                                               // return the result
}

// terminates the shared memeory tracking
void shmmgr_term (shmmgr_t *shmmgr) {
   int r;

   for (int i = 0; i < SHMMGR_REGION_MAX; i++) {           // loop through all the shared memory segment data
      if (shmmgr->key[i] != SHMMGR_UNALLOCATED) {          // find a key that indicates an allocation was made

         r = shmctl (shmmgr->shmid[i], IPC_RMID, NULL);    // remove/deallocate the shared memory
         if (r) {                                          // write an error and exit on failure
            dylib.fputs ("shmctl failed\n", stderr);
            exit (1);
         }
      }
   }
}

// returns a pointer to the value address 
void *shmmgr_map_and_return_ptr (shmmgr_t *shmmgr, int index) {
 
   int region = index / shmmgr->region_div;                      // determine which shared memory segment, page and entry corresponds to
   int page   = (index % shmmgr->region_div) / shmmgr->page_div; // this index
   int entry  = (index % shmmgr->region_div) % shmmgr->page_div;

   char *r = (char *) 0xf000;                                    // a bit of a hack. shared memory will by default be located here

   if (region != shmmgr->cur_region ||                           // determine if the current shared memory and page are the wrong ones
       page   != shmmgr->cur_page) {
                                                                 // they are...
      r = (char*) shmat (shmmgr->shmid[region], NULL, 0, page);  // change to the right shared memory segment and page
      if (r == (char*) -1) {                                     // write an error and exit on failure
         dylib.fputs ("shmat failed\n", stderr);
         exit (1);
      }

      shmmgr->cur_region = region;                               // save the current cached shared memory segment and page ids
      shmmgr->cur_page   = page;
   }

   r += entry * shmmgr->alloc_size;                              // calculate the address to be returned

   return r;                                                     // return the address
}

// frees an allocation
void shmmgr_free (shmmgr_t *shmmgr, int index) {

// int alloc_matrix_index = index / 8;                         // get the matrix index
   int alloc_matrix_index = index >> 3;                        // get the matrix index
// int bit = index % 8;                                        // calc the bit position
   int bit = index & 0x0007;                                   // calc the bit position
 
   char mask = ~(0x01 << bit);                                 // generate the mask, which is 11111111 except the bit to be unset

   shmmgr->alloc_matrix[alloc_matrix_index] =                  // unset the used flag
      shmmgr->alloc_matrix[alloc_matrix_index] & mask;  

   if (alloc_matrix_index < shmmgr->alloc_matrix_first_free) { // if this alloc matrix index is less than the current marked free
      shmmgr->alloc_matrix_first_free = alloc_matrix_index;    // position, then move it
   }
}  

// returns an index to the allocated storage
int shmmgr_alloc (shmmgr_t *shmmgr) {

   int r = -1;                                                  // default the return value

   for (int i = shmmgr->alloc_matrix_first_free;                // loop through the allocation table
//      i < shmmgr->alloc_max / 8; 
        i < shmmgr->alloc_max >> 3;
        i++) {                                          

      if (shmmgr->alloc_matrix[i] != SHMMGR_ALLOC_FULL) {       // test for an entry with a free position
         unsigned char m = ~ shmmgr->alloc_matrix[i];           // found one
         unsigned char n = 0x01;
         for (int j = 0; j < 8; j++) {                          // find the free one in the bit array
            if (m & n) {                                        // test if this is the free one
                                                                // it is...
//             r = i * 8 + j;                                   // capture the index as the return value
               r = (i << 3) + j;                                // capture the index as the return value

               int region = r / shmmgr->region_div;             // calculate which region this is in
 
               if (shmmgr->key[region] == SHMMGR_UNALLOCATED) { // allocated a shared memory segment if not currently allocated
                  char key_name[6];                             // generate a key name for the shared memory
                  strcpy (key_name, shmmgr->name);
                  char t[2];
                  t[0] = '0' + region;
                  t[1] = 0x00;
                  dylib.strcat (key_name, t);
                  shmmgr->key[region]   = ftok (key_name, 0);   // generate a key token
                  shmmgr->shmid[region] =                       // create the shared memory region
                     shmget 
                        (shmmgr->key[region], 
                         SHMMGR_REGION_ALLOC_SIZE, 
                         IPC_CREAT | IPC_EXCL); 
                  if (shmmgr->shmid[region] == -1) {            // test for failure
                     dylib.fputs ("shmget failed\n", stderr);   // print error and exit
                     exit (1);
                  }
               }

               shmmgr->alloc_matrix[i] =                        // mark the allocation in the matrix
                  shmmgr->alloc_matrix[i] | n; 

               shmmgr->alloc_matrix_first_free = i;             // set the next starting point

               break;                                           // exit the inner loop
            }
            n = n << 1;                                         // shift the bit, try again
         }
      }
      if (r >= 0) {                                             // if a free index is found, exit the outer loop
         break;
      }
   }

   return r;                                                    // return the index
}
