/* soundqueue.c
   This package initialzes the soundqueue.
   06/23/2023 initial version
   09/29/2023 added termination of sound
   10/21/2023 removed use of memory library
   10/22/2023 updated to use public sequencer methods
   11/03/2023 moved vdp polling to the seqeuncer
              added auto-initialzation of the soundqueue package
   10/25/2025 forced initial interrupt after adding sound to the queue
*/

#include <vdp.h>
#include <stdbool.h>
#include <stdlib.h>
#include <soundqueue.h>
#include <soundqueue_private.h>
#include <sounds.h>
#include <constants.h>
#include <sequencer.h>
#include <sequencer_private.h>
#include <dylib.h>

int h_beep = UNDEFINED;
int h_honk = UNDEFINED;
int h_tipi = UNDEFINED;

bool soundqueue_is_initialized = false;

soundqueue_info_t soundqueue_info;

#define SOUND_TONE_MAX_LEN 14
const char sound_tone_off_cmd[] = {0x04, TONE1_VOL | 0x0f, TONE2_VOL | 0x0f, TONE3_VOL | 0x0f, NOISE_VOL | 0x0f, 0x00};

void soundqueue_terminate () {
   soundqueue_stop ();
   while (!soundqueue_play_is_done ()) {
      sequencer_update ();
   }
}

void soundqueue_init () {
   if (!soundqueue_is_initialized) {
      soundqueue_info.buf_len = 32;
      soundqueue_info.buf_max_entries = (soundqueue_info.buf_len - sizeof (sound_tone_off_cmd)) / SOUND_TONE_MAX_LEN;
      soundqueue_info.count           = 0;
      soundqueue_info.current         = UNDEFINED;
      for (int i = 0; i < SOUND_ENTRIES_NUM; i++) {
         soundqueue_info.entry[i].in_use = false;
      }
   
      // add sound playing to the sequencer
      sequencer_add (soundqueue_play_update);
      // add atexit to shutdown sound
      dylib.atexit (soundqueue_terminate);
      soundqueue_is_initialized = true;
   }
}

void soundqueue_play_update () {

   int len;
   int i;
   int c;
   char *p;

   // handle only if there's sound being played
   if (soundqueue_info.current >= 0) {

      // update if the sound count value is zero
      if (SOUND_CNT == 0) {

         // add more sound to the buffer if more remains
         if (soundqueue_info.entry[soundqueue_info.current].pos < soundqueue_info.entry[soundqueue_info.current].len) {

            // set to write to the sound buffer in VDP RAM
            // VDP_SET_ADDRESS_WRITE (soundqueue_info.buffer[0].addr);
            // VDP_SET_ADDRESS_WRITE (soundqueue_info.buffer[soundqueue_info.entry[soundqueue_info.current].sbuf_id].addr);
            VDP_SET_ADDRESS_WRITE (VDP_SOUND_BUFFER_ADDR);

            c = 0;
            while ( (soundqueue_info.entry[soundqueue_info.current].pos < soundqueue_info.entry[soundqueue_info.current].len) && 
                    (c < soundqueue_info.buf_max_entries)) {

               //  calculate the next address of data to send
               p = soundqueue_info.entry[soundqueue_info.current].saddr + soundqueue_info.entry[soundqueue_info.current].pos;

               // determine the number of bytes to send, which are the number of bytes in the note, the 
               // note itself, and the duration
               len = (*p) + 2;

               // len f v f v f v n v d = 10; up to 25 = 250
               // len v v v v d         = 6            + 6   = 256

               // send the bytes
               for (i = 0; i < len; i++) {
                  VDPWD = *p;
                  p++;
               }

               // update the position info
               soundqueue_info.entry[soundqueue_info.current].pos += len;

               c++;
            }

            if (soundqueue_info.entry[soundqueue_info.current].pos < soundqueue_info.entry[soundqueue_info.current].len) {
               // terminate the note with sound off
               p = (char *) sound_tone_off_cmd;
               for (i = 0; i < sizeof (sound_tone_off_cmd); i++) {
                  VDPWD = *p;
                  p++;
               }
            }

            // set the sound pointer to the sound buffer in VDP RAM
            SET_SOUND_PTR (VDP_SOUND_BUFFER_ADDR);

            // set that the sound is to be played from VDP RAM
            SET_SOUND_VDP ();

            // start sound with this new data
            START_SOUND ();

            VDP_INT_POLL;
         } else {
            // set the current entry to undefined, and then attempt to locate a preempted sound entry and continue playing that
            soundqueue_info.current = UNDEFINED;
            for (i = 0; i < SOUND_ENTRIES_NUM; i++) {
               if (soundqueue_info.entry[i].in_use && (soundqueue_info.entry[i].pos < soundqueue_info.entry[i].len)) {
                  soundqueue_info.current = i;
                  break;
               }
            }
         }
      }
   }
}
