/* stat.c
   This package provide filesystem file status.
   Change history
   05/21/2025 initial version
   05/28/2025 updated to use the file types provided by dirent which ultimately come from the TI DSRs. 0 to 5 are file types,
              6 is a directory
   06/02/2025 added dylib references
   06/10/2025 updated S_IS_REG to return true for any file type
*/

#include <sys/stat.h>
#include <dirent.h>
#include <libgen.h>
#include <string.h>
#include <conversion.h>
#include <dylib.h>

#define STAT_FILE_TIDISFIX  0x0001    // TI's value for text files stored in fixed record length (not used/supported in Unix99)
#define STAT_FILE_TIDISVAR  0x0002    // TI's value for text files stored in variable length records. Unix99 uses this format for text files
#define STAT_FILE_TIINTFIX  0x0003    // TI's value for internal/fixed (binary) files. Unix99 uses this format for binary files
#define STAT_FILE_TIINTVAR  0x0004    // TI's value for internal/variable (binary) files (not used/supported in Unix99)
#define STAT_FILE_TIPROGRAM 0x0005   // TI's value for program files (not used/supported in Unix99)
#define STAT_DIR_TI         0x0006    // TI's value for directory returned by directory catalog

int S_ISDIR (unsigned int st_mode) {
   return st_mode == STAT_DIR_TI;
}

int S_ISREG (unsigned int st_mode) {
   return (st_mode >= STAT_FILE_TIDISFIX) && (st_mode <= STAT_FILE_TIPROGRAM);
}

int S_ISTIDISVAR (unsigned int st_mode) {
   return st_mode == STAT_FILE_TIDISVAR;
}

int S_ISTIINTFIX (unsigned int st_mode) {
   return st_mode == STAT_FILE_TIINTFIX;
}

int stat (const char *path, struct stat *buf) {

   int r = 1;

   struct dirent *e;
   DIR *d = opendir (path);
   if (d) {
      buf->st_size = 0;
      buf->st_mode = STAT_DIR_TI;
      closedir (d);
      r = 0;
   } else {
      char *dname = dirname ((char *) path);
      char *bname = basename ((char *) path);
      d = opendir (dname);
      while ((e = readdir (d))) {
         if (!dylib.strcmp (bname, e->d_name)) {
            buf->st_size = ftol (e->d_sectors) * (long) 256;
            buf->st_mode = ftol (e->d_type);
            r = 0;
            break;
         }
      }
      closedir (d);
   }

   return r;
}
