/* statfs.c
   This method profiles file system status.
   change log:
   06/23/2023 initial version
   08/15/2023 updated for unix paths and to construct final path from cwd + provided path
   11/15/2023 updated to collect the DSR's path
   11/16/2023 removed dead code
   11/21/2023 updated to use FILENAME_MAX
   12/10/2023 removed reference to string_ext.h
   03/01/2024 modified to use dylib
   03/07/2024 updated to use dylib's getcwd
   01/10/2025 updated to use more dylib methods
*/

#include <string.h>
#include <dirent.h>
#include <statfs.h>
#include <unistd.h>
#include <unistd_private.h>
#include <dylib.h>

int statfs (const char *path, struct statfs *buf) {

   int r = -1;
   char *p;

   // construct the full path of the path provided
   char cwd[FILENAME_MAX];
   p = dylib.getcwd (cwd, sizeof (cwd));
   if (!p) dylib.strcpy (cwd, "");
   bool final_path_ok;
   char final_path[FILENAME_MAX];
   dylib.generate_unix_path (cwd, (char *) path, &final_path_ok, final_path);

   // if the full path was constructed properly then perform the statfs
   if (final_path_ok) {
      // the underlying DSR for the root fs is implemented in generate_final_path() so this call will result in a fully 
      // qualified ti DSR path, that must be reconverted back to a fully qualified unix path before being sent to opendir()
      char final_ti_path[FILENAME_MAX];
      dylib.generate_final_path (final_ti_path, final_path, true);
      dylib.strcpy (final_path, "/");
      p = dylib.strchr (final_ti_path, '.');
      *p = 0x00;
      dylib.strcat (final_path, final_ti_path);

      DIR *d = opendir (final_path);        // open the directory at the top level dsr path
      if (d) {                              // check if opened
         dylib.strcpy (buf->volume, d->volume);   // got it, copy the volume name
         buf->f_blocks = d->total_sectors;  // capture the total number of sectors
         buf->f_bfree  = d->free_sectors;   // capture the free sectors
         closedir (d);                      // close the director
         r = 0;                             // set the return value to success
      }
   }

   return r; // return the result
}
