/* system.c
   This method provides an interface to execute a complex set of commands formatted identically to shell commands.
   change log:
   11/03/2023 initial version, integrating code originally included in the shell command.
   11/15/2023 removed use of get_root_fs_
   11/16/2023 updated documentation
   12/10/2023 removed reference to string_ext.h
   12/10/2023 added reference to conversion.h
   02/11/2024 updated to test for INTERNAL rather than PROGRAM file
   02/12/2024 removed debug statements
              updated to use file_is_executable rather than get_dsr_status
   03/01/2024 modified to use dylib
   03/06/2024 added call to exit if commands were processed successfully
   06/16/2025 updated exit reference
   06/23/2025 added checking on limit of number of arguments
              added checking arg length out of quotes
              added checking arg length in quotes
   06/24/2025 added test for valid redirect filename. This method should now be bullet proof.
*/

#include <system_private.h>
#include <string.h>
#include <conversion.h>
#include <constants.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdio_private.h>
#include <unistd.h>
#include <unistd_private.h>
#include <soundqueue.h>
#include <cache_private.h>
#include <dylib.h>
#include <stdlib.h>

#define MAX_ARGC     7
#define MAX_ARG_LEN 32
// value type
typedef struct {
   char v[MAX_ARG_LEN];
} v_t;

// command type
typedef struct {
   v_t argv[MAX_ARGC];
   int argc;
   v_t pin;
   v_t pout;
   v_t redirout;
   bool have_redirect;
   bool redirect_is_append;
} cmd_t;

// initialize the argument structure
void init_args (cmd_t *cmd) {
   for (int i = 0; i < MAX_ARGC; i++) {   // loop through all args
      dylib.strcpy (cmd->argv[i].v, "");  // initialize this arg
   }
   cmd->argc = 0;                         // set the arg count to zero
}

// initialize the command
void init_cmd (cmd_t *cmd) {
   init_args (cmd);                       // initialize the arguments
   dylib.strcpy (cmd->pin.v, "");         // initialize the stdin value
   dylib.strcpy (cmd->pout.v, "");        // initialize the stdout value
   dylib.strcpy (cmd->redirout.v, "");    // initialize the redirect value
   cmd->have_redirect = false;            // initialize the redirect flag
}

// adds a command, indicating if the command was actually added and has errors
void add_cmd (cmd_t *cmd, bool *did_add_cmd, bool *has_errors) {

   // initialize the return values
   *did_add_cmd = false;
   *has_errors = false;

   // count the number of arguments
   for (int i = 0; i < MAX_ARGC; i++) {
      if (dylib.strlen (cmd->argv[i].v)) {
         cmd->argc = i + 1;
      }
   }

   // detect whether a command is present, setting the add flag and has_errors flag
   if (dylib.strlen (cmd->argv[0].v)) {
      *did_add_cmd = true;
   } else {
      *has_errors = (dylib.strlen (cmd->pin.v) > 0) || (dylib.strlen (cmd->pout.v) > 0) || (dylib.strlen (cmd->redirout.v) > 0);
   }
   *has_errors |= cmd->have_redirect && (dylib.strlen (cmd->redirout.v) == 0);

   if (*did_add_cmd) {
      // capture stdin and stdout, and redirection
      char *pstdin, *pstdout;

      if (dylib.strlen (cmd->pin.v)) {
         pstdin = cmd->pin.v;
      } else {
         pstdin = NULL;
      }
   
      if (dylib.strlen (cmd->pout.v)) {
         pstdout = cmd->pout.v;
      } else {
         if (dylib.strlen (cmd->redirout.v)) {
            pstdout = cmd->redirout.v;
            if (file_is_valid_path (pstdout)) {
               if (cmd->redirect_is_append) {
                  dylib.strcat (cmd->redirout.v, " a");
               } else {
                  dylib.strcat (cmd->redirout.v, " w");
               }
            } else {
               *has_errors = true;
            }
         } else {
            pstdout = NULL;
         }
      }
   
      // set up the argv structure
      char *argv[MAX_ARGC];
      for (int i = 0; i < MAX_ARGC; i++) {
         argv[i] = cmd->argv[i].v;
      }
   
      // capture the unaliased command and replace the alias
      const char *long_cmd = get_unaliased_cmd (argv[0]);
      if (long_cmd) {
         argv[0] = (char *) long_cmd;
      }

      // confirm dsr_status is defined and that the command is a program type; if so then add it
      if (!*has_errors) {
         if (file_is_executable (argv[0])) {
            proc_add (pstdin, pstdout, NULL, cmd->argc, (const char **) argv);
         } else {
            // inidicate that the command wasn't added and has errors
            *did_add_cmd = false;
            *has_errors  = true;
         }
      } else {
         *did_add_cmd = false;
      }
   }
}

// process the commands from the input string
// general and representative command sequences are as follows:
// cmd1
// cmd1 p1 p2
// cmd1 p1 p2 | cmd
// cmd1 p1 p2 > f1
// cmd1 ; cmd2

bool process_cmds (char *s) {

   cmd_t cmd;
   int len;
   int pipeid;
   int i;

   init_cmd (&cmd);
   pipeid = 0;
   bool did_add_cmds     = false;
   bool have_errors      = false;
   bool dac, he;
   bool in_quoted_string = false;

   char *p;

   // loop through all of s
   len = dylib.strlen (s);
   for (i = 0; i < len; i++) {
      if (in_quoted_string) {                               // test for in quote
         switch (s[i]) {
            case '\"':                                      // handle quote termination
               if (dylib.strlen (cmd.argv[cmd.argc].v)) {
                  cmd.argc++;
               }
               in_quoted_string = false;
               break;
            default:                                        // handle text between quotes
               if (cmd.have_redirect) {
                  if (dylib.strlen (cmd.redirout.v) < MAX_ARG_LEN - 1) {
                     strncat (cmd.redirout.v, &s[i], 1);
                  } else {
                     have_errors = true;
                  }
               } else {
                  if (dylib.strlen (cmd.argv[cmd.argc].v) < MAX_ARG_LEN - 1) {
                     strncat (cmd.argv[cmd.argc].v, &s[i], 1);
                  } else {
                     have_errors = true;
                  }
               }
         }
      } else {                                              // not in a quote
         switch (s[i]) {
            case '\"':                                      // start handling quoted string
               in_quoted_string = true;
               break;
            case ';':                                       // handle compound command
               add_cmd (&cmd, &dac, &he);
               did_add_cmds |= dac;
               have_errors  |= he;
               init_cmd (&cmd);
               break;
            case '|':                                       // handle piped command
               dylib.strcpy (cmd.pout.v, "/proc/pipe");
               dylib.strcat (cmd.pout.v, dylib.int2str (pipeid));
               dylib.strcat (cmd.pout.v, " w");             // append the file mode
               add_cmd (&cmd, &dac, &he);
               did_add_cmds |= dac;
               have_errors  |= he;
               init_args (&cmd);
               p = dylib.strtok (cmd.pout.v, " ");          // remove the appended file mode
               p = dylib.strtok (NULL, " ");
               *p = 0x00;
               dylib.strcpy (cmd.pin.v, cmd.pout.v);
               dylib.strcpy (cmd.pout.v, "");
               pipeid++;
               break;
            case '>':                                       // handle redirected stdout to file
               cmd.have_redirect = true;
               if (s[i + 1] == '>') {                       // if there's a second redirect symbol then this is an append
                  cmd.redirect_is_append = true;            // set this as append
                  i++;                                      // skip over the second redirect symbol
               } else {
                  cmd.redirect_is_append = false;           // set this as non-append
               }
               break;
            case ' ':                                       // handle space between commands and arguments
               if (dylib.strlen (cmd.argv[cmd.argc].v)) {
                  cmd.argc++;
                  if (cmd.argc >= MAX_ARGC) {
                     have_errors = true;
                  }
               }
               break;
            default:                                        // collect everything else
               if (cmd.have_redirect) {
                  if (dylib.strlen (cmd.redirout.v) < MAX_ARG_LEN - 1) {
                     strncat (cmd.redirout.v, &s[i], 1);
                  } else {
                     have_errors = true;
                  }
               } else {
                  if (dylib.strlen (cmd.argv[cmd.argc].v) < MAX_ARG_LEN - 1) {
                     strncat (cmd.argv[cmd.argc].v, &s[i], 1);
                  } else {
                     have_errors = true;
                  }
               }
               break;
         }
      }

      if (have_errors) {
         break;
      }
   }

   if (in_quoted_string) {               // make sure quoted strings have been terminated
      have_errors = true;
   }
   if (!have_errors) {
      add_cmd (&cmd, &dac, &he);         // handle the last command
      did_add_cmds |= dac;
      have_errors  |= he;
   }

   if (have_errors) {                    // handle errors if present
      proc_cache_init ();
      did_add_cmds = false;
   }

   return did_add_cmds;                  // return result
}

int system (const char *command) {
   int r = UNDEFINED;
   if (process_cmds ((char *) command)) {
      r = 0;
      exit (0);                    // this will launch the commands
   }

   return r;
}
