/* vsprintf.c
   This method provides the core formatted printing capability used by fprintf, printf, and sprintf.
   change log:
   06/23/2023 initial version
   12/10/2023 removed reference to string_ext.h
   02/24/2024 cleanup
   03/01/2024 updated to use dylib
   06/07/2025 updated to use ptoa_upper
   06/16/2025 corrected hex (%X and %x) display to support width, size, etc.
   10/13/2025 removed dylib uint2str with direct call
*/

#include <string.h>
#include <constants.h>
#include <conversion.h>
#include <stdarg.h>
#include <dylib.h>

static char *xprint_rwp;

static inline void vsprintf_add_one_char (int c) {
   *xprint_rwp = (char) c;
   xprint_rwp++;
}

// helper function for floating point values
static int vsprintf_format_print_fp (char *s, int zero, int width, int left, int precis, int plus) {

   char buffer[64];
   char *w;
   char *p;
   char *dp;
   int neg;
   int len_i;
   int len_f;
   int len_t;

   p = s;

   dp  = strstr (p, ".");
   if (dp) {
      *dp = 0; // null terminate
      dp++;    // move to the fractional digits
   }

   if (*p == '-') {
      neg  = 1;  // capture that this value is negative
      plus = 0;  // dont need the plus to be used
      p++;       // move past the sign
   } else {
      neg = 0;
   }

   len_i = dylib.strlen (p);
   if (dp) len_f = dylib.strlen (dp); else len_f = 0;
   len_t = neg + plus + len_i + (precis > 0) + precis;
/* total length=
           ^     ^        ^         ^            ^
           -sign--   int portion    |        # precis digits
                           whether a dp will 
                               be printed

   length of the actual precision digits is unimportant here
*/

   w = buffer;
   
   char pad[2] = {0, 0};
   int i;

   if (zero) pad[0] = '0'; else pad[0] = ' ';
   for (i = 0; i < width - len_t; i++) {
      *w = pad[0];
      w++;
   }

   if (neg) {
      *w = '-';
      w++;
   }

   if (plus) {
      *w = '+';
      w++;
   }

   for (i = 0; i < len_i; i++) {
      *w = *p;
      w++;
      p++;
   }

   if (precis) {
      *w = '.';
      w++;

      if (len_f > precis) len_f = precis;
      for (i = 0; i < len_f; i++) {
         *w = *dp;
         w++;
         dp++;
      }
   
      for (i = 0; i < precis - len_f; i++) {
         *w = '0';
         w++;
      }
   }

   *w = 0;

   w = buffer;
   for (i = 0; i < dylib.strlen (buffer); i++) {
      vsprintf_add_one_char (*w);
      w++;
   }

   return dylib.strlen (w);
}

// helper function to put the string out - string will be modified and is
// expected to be 256 chars in length
static int xprintf_format_print (char *s, int zero, int width, int left, int precis, int plus) {
    int cnt = 0;
    int i = dylib.strlen (s);
    int pad = ' ';
    if (zero) pad = '0';

    if ((precis > 0) && (i >= precis)) {
        i = precis;
        if (plus) --i;
    }
    if ((plus) && (width > 0)) --width;
    if (width <= 0) {
        if (plus) {
            vsprintf_add_one_char ('+');
            ++cnt;
        }
        for (int idx=0; idx<i; ++idx) {
            vsprintf_add_one_char (*(s++));
            ++cnt;
        }
    } else {
        if (left) {
            if (plus) {
                vsprintf_add_one_char ('+');
                ++cnt;
            }
            for (int idx=0; idx<i; ++idx) {
                vsprintf_add_one_char (*(s++));
                ++cnt;
            }
            for (int idx=0; idx<width-i; ++idx) {
                vsprintf_add_one_char (' ');
                ++cnt;
            }
        } else {
            if ((plus)&&(zero)) {
                vsprintf_add_one_char ('+');
                plus = 0;
                ++cnt;
            }
            for (int idx=0; idx<width-i; ++idx) {
                vsprintf_add_one_char (pad);
                ++cnt;
            }
            if (plus) {
                vsprintf_add_one_char ('+');
                ++cnt;
            }
            for (int idx=0; idx<i; ++idx) {
                vsprintf_add_one_char (*(s++));
                ++cnt;
            }
        }
    }
    return cnt;
}

int vsprintf (char * restrict str, const char * restrict format, va_list argp) {
    char xprintfbuf[16];
    int iv;
    unsigned int uv;
    char *sv;
    char *svt;
    int cnt = 0;
    long lv;
    unsigned long ul;
    double d;
    void *vp;

    int zero;
    int width;
    int left;
    int plus;
    int ucase;
    int precis;
    int have_p;
    int longv;
    int done;

    xprint_rwp = str;

    while (*format) {
        if (*format != '%') {
            vsprintf_add_one_char (*format);
        } else {
            ++format;
            zero   = 0;
            width  = 0;
            left   = 0;
            plus   = 0;
            ucase  = 0;
            precis = 0;
            have_p = 0;
            longv  = 0;
            done   = 0;
            while (!done) {
                switch (*format) {
                    case '-': left    =  1; break;
                    case '+': plus    =  1; break;
                    case '.': precis  = -1; have_p = 1; break;
                    case 'l': longv   =  1; break;
                    case '0':
                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9': 
                        if (precis) {
                            if (precis == -1) {
                                precis = (*format)-'0';
                            } else {
                                precis = (precis*10) + ((*format)-'0');
                            }
                        } else {
                            if (width == 0) { 
                                if (*format == '0') {
                                    // this i a zero pad specifier
                                    zero=1;
                                } else {
                                    width = (*format)-'0';
                                }
                            } else {
                                width = (width*10) + ((*format)-'0');
                            }
                        }
                        break;

                    case 'c':   // char
                       iv = va_arg(argp, int);
                       xprintfbuf[0] = iv;
                       xprintfbuf[1] = 0x00;
                       cnt += xprintf_format_print (xprintfbuf, 0, width, left, precis, 0);
                       done = 1;
                       break;

                    case 'i':   // decimal
                    case 'd':   // decimal
                        if (longv) {
                           lv = va_arg (argp, long);
                           sv = (char *) ltoa (lv);
                           cnt += xprintf_format_print (sv, zero, width, left, precis, plus);
                           done = 1;
                        } else {
                           iv = va_arg (argp, int);
                           sv = dylib.int2str (iv);
                           cnt += xprintf_format_print (sv, zero, width, left, precis, plus);
                           done = 1;
                        }
                        break;

                    case 'u':   // unsigned decimal
                        if (longv) {
                           ul = va_arg (argp, unsigned long);
                           sv = (char *) ultoa (ul); 
                           cnt += xprintf_format_print (sv, zero, width, left, precis, plus);
                           done = 1;
                        } else {
                           uv = va_arg(argp, unsigned int);
                           sv = uint2str (uv);
                           cnt += xprintf_format_print (sv, zero, width, left, precis, plus);
                           done = 1;
                        }
                        break;

                    case 's':   // string
                        sv = va_arg(argp, char*);
                        cnt += xprintf_format_print(sv, 0, width, left, precis, 0);
                        done = 1;
                        break;

                    case 'X':   // upper case hex
                        if (longv) {
                           ul = va_arg (argp, unsigned long);
                           sv = (char *) ptoa_upper (&ul, 4);
                           svt = strpbrk (sv, "123456789ABCDEF");
                           if (svt) {
                              sv = svt;
                           }
                           cnt += xprintf_format_print (sv, zero, width, left, precis, plus);
                           done = 1;
                        } else {
                           uv = va_arg(argp, unsigned int);
                           sv = (char *) ptoa_upper (&uv, 2);
                           svt = strpbrk (sv, "123456789ABCDEF");
                           if (svt) {
                              sv = svt;
                           }
                           cnt += xprintf_format_print (sv, zero, width, left, precis, plus);
                           done = 1;
                        }
                        break;
                    case 'x':   // lower case hex
                        if (longv) {
                           ul = va_arg (argp, unsigned long);
                           sv = (char *) ptoa_lower (&ul, 4);
                           svt = strpbrk (sv, "123456789abcdef");
                           if (svt) {
                              sv = svt;
                           }
                           cnt += xprintf_format_print (sv, zero, width, left, precis, plus);
                           done = 1;
                        } else {
                           uv = va_arg(argp, unsigned int);
                           sv = (char *) ptoa_lower (&uv, 2);
                           svt = strpbrk (sv, "123456789abcdef");
                           if (svt) {
                              sv = svt;
                           }
                           cnt += xprintf_format_print (sv, zero, width, left, precis, plus);
                           done = 1;
                        }
                        break;
/* libti99:
                    case 'X':   // uppercase hex
                        ucase = 1;
                        // fall through

                    case 'x':   // hex (a little inefficient..)
                        uv = va_arg(argp, unsigned int);
                        iv = (uv&0xf000)>>12;
                        if (iv>9) iv+=7;
                        xprintfbuf[0]=iv+'0';
                        iv = (uv&0xf00)>>8;
                        if (iv>9) iv+=7;
                        xprintfbuf[1]=iv+'0';
                        iv = (uv&0xf0)>>4;
                        if (iv>9) iv+=7;
                        xprintfbuf[2]=iv+'0';
                        iv = (uv&0xf);
                        if (iv>9) iv+=7;
                        xprintfbuf[3]=iv+'0';
                        xprintfbuf[4]='\0';
                        if (!zero) {
                            // remove leading zeros
                            while (xprintfbuf[0]=='0') {
                                // my memcpy is safe in this direction only...
                                dylib.memcpy(&xprintfbuf[0], &xprintfbuf[1], 4);  // includes NUL
                            }
                            if (xprintfbuf[0] == '\0') {
                                xprintfbuf[0]='0';
                                xprintfbuf[1]='\0';
                            }
                        }
                        if (!ucase) {
                            // make lowercase
                            for (int idx=0; idx<4; ++idx) {
                                if ((xprintfbuf[idx]>='A')&&(xprintfbuf[idx]<='F')) {
                                    xprintfbuf[idx]+=32;  // make lowercase
                                }
                            }
                        }
                        cnt += xprintf_format_print(xprintfbuf, 0, width, left, precis, 0);
                        done = 1;
                        break;
*/
                    case 'o':   // octal (roughly same as hex)
                        uv = va_arg(argp, unsigned int);
                        iv = (uv&0x8000)>>15;
                        xprintfbuf[0]=iv+'0';
                        iv = (uv&0x7000)>>12;
                        xprintfbuf[1]=iv+'0';
                        iv = (uv&0x0e00)>>9;
                        xprintfbuf[2]=iv+'0';
                        iv = (uv&0x01c0)>>6;
                        xprintfbuf[3]=iv+'0';
                        iv = (uv&0x0038)>>3;
                        xprintfbuf[4]=iv+'0';
                        iv = (uv&0x0007);
                        xprintfbuf[5]=iv+'0';
                        xprintfbuf[6]='\0';
                        if (!zero) {
                            // remove leading zeros
                            while (xprintfbuf[0]=='0') {
                                // my memcpy is safe in this direction only...
                                dylib.memcpy(&xprintfbuf[0], &xprintfbuf[1], 6);  // includes NUL
                            }
                            if (xprintfbuf[0] == '\0') {
                                xprintfbuf[0]='0';
                                xprintfbuf[1]='\0';
                            }
                        }
                        cnt += xprintf_format_print(xprintfbuf, 0, width, left, precis, 0);
                        done = 1;
                        break;

                    case '%':   // percent sign
                        vsprintf_add_one_char ('%');
                        ++cnt;
                        done = 1;
                        break;

                    // probably useful...
                    //case 'e':   // exp float (no support)

                      case 'f':   // float
                        d = va_arg (argp, double);
                        sv = (char *) ftoa (d); 
                        if (!have_p) precis = 6; // set a default precision if not provided in the format
                        cnt += vsprintf_format_print_fp (sv, zero, width, left, precis, plus);
                        done = 1;
                        break;

                    case 'p':    // void pointer
                        vp = va_arg (argp, void *);
                        sv = (char *) ptoa_upper (vp, width);
                        cnt += xprintf_format_print (sv, zero, width, left, precis, plus);
                        done = 1;
                        break;

                    case '\0':
                        // error - end of string
                        --format;
                        done = 1; // it may be an error but need to break out
                        break;

                    default:
                        vsprintf_add_one_char (*format);
                        ++cnt;
                        done = 1;
                        break;
                }
                ++format;
            }
            --format;  // make up for the extra increment
        }
        ++format;
    }

    *xprint_rwp = 0;

    return cnt;
}
