/* kernel_exit.c
   This file includes the methods associated with exiting a program.
   change log:
   06/23/2023 initial version
   07/27/2023 added calls to _Exit to ensure that system caches in 0x2000
              are reset appropriately (set character definitions as changed
              and number of processes to zero.
   08/11/2023 Removed call to invalidate fonts; added call to cache_invalidate
   09/02/2023 modified to use plugin for get_rom_start_addr
   09/08/2023 updated to look for UNIX99
   11/02/2023 updated for system_private.h name changes
   11/16/2023 updated to call the cached rom by addr
   02/24/2024 modified to use cache
   03/01/2024 updated exit to launch the next program in the chain, which will be a user app, login, or the shell
   01/06/2025 corrected _Exit to reset sams
   01/07/2025 removed call to cache invalidate
   01/17/2025 migrated all methods other than exit() from this file
   06/11/2025 added test to ensure no trailing key holds from this to next program
   06/16/2025 renamed from exit.c; now executable only within the kernel (rom)
*/

#include <stdlib.h>
#include <unistd.h>
#include <stdlib_private.h>
#include <unistd_private.h>
#include <cache_private.h>
#include <kscan.h>

void kernel_exit (int status) {

   // call all the registered atexit methods
   for (int i = cache.atexit_info.count - 1; i >= 0; i--) {
      cache.atexit_info.method[i] ();
   }

   // ensure the keyboard is clear -- no trailing key holds from this program to the next
   while (kscan (5) != 0xff);

   // start the next ea5 program if so queued
   proc_exec_next_ea5 ();

   // assess whether the login program or shell needs to be started:
   // when there's no session id then noone is logged in
   if (getsessionid () == 0) {
      login_cache_init ();                // since there's no session id, reset the user to root ahead of the login program running
      exec_uea5 ("/bin/login");           // execute the login program
   } else {
      exec_uea5 (get_user_shell_path ()); // a session is active, so set the next program to run as the shell
   }
   _Exit (status);

}
