/* kernel_proc_get_ea5_params.c
   This method retrieves the command, arguments, stdin/out/err and cwd for the EA5 program about to be executed,
   if it is defined in the process list.
   change log:
   06/23/2023 initial version
   08/05/2023 added cwd processing
   08/11/2023 modified to use system cache for proc list
   09/08/2023 updated references to UNIX99 INITD
   11/21/2023 removed cwd param and reading of cwd
              updated to use FILENAME_MAX
   12/10/2023 removed reference to string_ext.h
              added reference to stdlib
   02/24/2024 broad changes for cache and dylib
   02/27/2024 increased use of dylib
   01/06/2024 updated definitions for std file names
   01/07/2025 updated to use text files riding on binary
   02/22/2025 renamed to kernel*
*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd_private.h>
#include <cache_private.h>
#include <conversion.h>

void kernel_proc_get_ea5_params (char *stdin_fn, char *stdout_fn, char *stderr_fn, int *argc, char *argv[]) {
   
   int i;

   // if there are processes to be executed then capture the process info for this one
   if (cache.proc_list.count) {

      // open the process file
      char tpath[FILENAME_MAX];
      FILE *f = fopen (proc_get_pid_path (cache.proc_list.id[0], tpath), "rb");

      // get the number of arguments
      char cs[4];
      fgets (cs, sizeof (cs), f);
      *argc = atoi (cs);

      // read the arguments
      for (i = 0; i < *argc; i++) {
         fgets (argv[i], 32, f);
         argv[i][strcspn (argv[i], "\r\n")] = 0;                    // remove CR/LF
      }

      // read the stdin file--will be blank if none set
      fgets (stdin_fn, FILENAME_MAX, f);
      stdin_fn[strcspn (stdin_fn, "\r\n")] = 0;                     // remove CR/LF

      // read the stdout file--will be blank if none was set
      fgets (stdout_fn, FILENAME_MAX, f);
      stdout_fn[strcspn (stdout_fn, "\r\n")] = 0;                   // remove CR/LF

      // read the stderr file--will be blank if none was set
      fgets (stderr_fn, FILENAME_MAX, f);
      stderr_fn[strcspn (stderr_fn, "\r\n")] = 0;                   // remove CR/LF

      // close the process file
      fclose (f);
 
      // remove this process id from the list by sliding the remaining process ids left
      cache.proc_list.count--;
      for (i = 0; i < cache.proc_list.count; i++) {
         cache.proc_list.id[i] = cache.proc_list.id[i + 1];
      }
   } else {
      // no process information--this only occurs after the UNIX99 INITD has detected no processes remain to be executed and has
      // launched the shell, and without a process file. Not ideal, but INITD living in 8KB ROM limits what can be done.
      strcpy (stdin_fn, "");
      strcpy (stdout_fn, "");
      strcpy (stderr_fn, "");
      *argc = 0;
   }
}
