/* kernel_shmget.c
   This method creates and/or provides an existing shared memory segment, returning an id to it.
   Note that the memory segment created is not initialized. 
   change history
   07/11/2025 initial version
   07/14/2025 updated to lock the allocated regions
*/

#include <conversion.h>
#include <string.h>
#include <sys/shm.h>
#include <sams_private.h>
#include <cache_private.h>

int kernel_shmget (key_t key, long size, int shmflg) {
   int r     = -1;
   int shmid = -1;

   int i;

   for (i = 0; i < SHM_ENTRIES_MAX; i++) {
      if (cache.shm.entry[i].in_use && cache.shm.entry[i].key == key) {
         shmid = i;
         break;
      }
   }

   if (shmid == -1) {
      for (i = 0; i < SHM_ENTRIES_MAX; i++) {
         if (!cache.shm.entry[i].in_use) {
            shmid = i;
            break;
         }
      }
   }

   if (shmid != -1) {

      if (!cache.shm.entry[shmid].in_use &&       // verify not in use
          (shmflg & IPC_CREAT) &&                 // verify intent to create
          (size > 0 && size <= (long) 196608)) {  // verify the requested size is legal

         int region;

         for (region = 0; region < SHM_REGION_MAX; region++) {
            cache.shm.entry[shmid].region[region] = -1;
         }
   
         int regions = (size - (long) 1) / (long) 24576 + (long) 1;
         char name[FILENAME_MAX];
         for (int region = 0; region < regions; region++) {
            strcpy (name, uint2hex (key));
            strcat (name, "_");
            strcat (name, uint2hex (region));
            sams_find_cache (name);
            cache.shm.entry[shmid].region[region] = cache.sams.region_in_use;        
            sams_lock_region (cache.shm.entry[shmid].region[region]);
         }
   
         sams_restore_proc_pages ();
         cache.shm.entry[shmid].key    = key;
         cache.shm.entry[shmid].in_use = true;
         r                             = shmid;
      } else {
         if (key == cache.shm.entry[shmid].key) {
            if (!((shmflg & IPC_CREAT) && (shmflg & IPC_EXCL))) {
               r = shmid;
            }
         }
      }
   }

   return r;
}
