/* life.c
   This program implements the game of life. To boost performance, the use of array addressing's multiplies were reduced, and
   the universe was expanded two rows longer and two columns wider, with the data offset by +1, +1. This eliminates a rather
   painful test to verify coordinates generated in the neighbor test are valid. That latter change reduced execution time by > 60%.
   change history
   05/17/2025 initial version
   06/02/2025 added call to reset the screen saver
   06/05/2025 handle signals
   06/07/2025 updated to use dylib
*/

#include <stdlib.h>
#include <stdio.h>
#include <console.h>
#include <console_private.h>
#include <unistd.h>
#include <dylib.h>

#define HEIGHT_MAX 32
#define WIDTH_MAX 82

long games;                                                        // number of games
int cycles;                                                        // number of cycles per game
int w;                                                             // width
int h;                                                             // height
unsigned int s = 0xace1;                                           // random seed
unsigned char univ[HEIGHT_MAX][WIDTH_MAX];                         // the universe - the universe is contained in 1..y,1..x
unsigned char new[HEIGHT_MAX][WIDTH_MAX];                          // updated universe
const unsigned int rand_max_half = (unsigned int) RAND_MAX / 2;    // half of the max of random
const char cell_visual[2] = {' ', '#'};                            // visualization parameter

// draws the universe
void universe_draw () {
                                                                   // this code is a bit complex, attempting to avoid the array
                                                                   // address computations
   unsigned char b[WIDTH_MAX + 2];                                 // display row
   int x, y;                                                       // position
   unsigned char *bp;                                              // pointer to buffer
   unsigned char *up;                                              // pointer to current position in universe

   // loop through the universe
   for (y = 1; y <= h; y++) {                                      // loop through all rows
      up = &univ[y][1];                                            // capture universe position on first column in row y
      bp = b;                                                      // capture the buffer first position
      for (x = 1; x <= w; x++) {                                   // loop through all columns
         *bp = cell_visual[*up];                                   // convert to displayable characters
         up++;                                                     // increment the universe position
         bp++;                                                     // increment the buffer position
      }
      console_write_raw (y - 1, 0, (char *) b, w);                 // write the row
   }
}
 
// update the universe with one cycle
void universe_cycle () {
   int x, y, x1, y1, n;                                            // positions

   int offset;                                                     // offset within new and univ
   unsigned char *np, *up;                                         // pointers within new and univ
   unsigned char *ip;

   for (y = 1; y <= h; y++) {                                      // loop through all rows
      offset = y * WIDTH_MAX + 1;                                  // calculate the offset within new and univ for this row
      up = (unsigned char *) univ + offset;                        // calculate the pointer to the current row in univ
      np = (unsigned char *) new + offset;                         // calculate the pointer to the current row in new
      for (x = 1; x <= w; x++) {                                   // loop through all columns
         n = 0;                                                    // initialize n
         for (y1 = y - 1; y1 <= y + 1; y1++) {                     // loop through the neighbors y
            ip = (unsigned char *) univ + y1 * WIDTH_MAX + x - 1;; // calculate the starting address in univ
            for (x1 = x - 1; x1 <= x + 1; x1++) {                  // loop through the neighbors x
               n += *ip;                                           // add in this position in the universe; note that since the
                                                                   // universe data is offset +1, +1 in univ, no need to test for
                                                                   // validity of the position. This is a big performance boost.
               ip++;                                               // move to the next position
            }
         }
         n -= *up;                                                 // subtract the current position value from the neighbor check
         *np = (n == 3 || (n == 2 && *up));                        // update the new universe value
         np++;                                                     // increment np
         up++;                                                     // increment up
      }
   }
   up = (unsigned char *) univ + WIDTH_MAX + 1;                    // calculate the first data position in the universe
   np = (unsigned char *) new + WIDTH_MAX + 1;                     // calculate the first data position in the new universe
   for (y = 1; y <= h; y++) {                                      // copy the new array values to univ
      memcpy (up, np, w);                                          // copy the row data from new to univ
      up += WIDTH_MAX;                                             // move to the next univ row's data
      np += WIDTH_MAX;                                             // move to the next new row's data
   }
}
 
// initializes the universe
void universe_init () {
   int x, y;
   memset (univ, 0, sizeof (univ));
   for (y = 1; y <= h; y++) {
      for (x = 1; x <= w; x++) {
         univ[y][x] = rand () < (unsigned int) rand_max_half ? 1 : 0;
      }
   }
}

// starts game play
int game_play (int cycles) {
   int r = 0;
   universe_init ();
   for (int i = 0; i < cycles; i++) {
      universe_cycle ();
      universe_draw ();
      if (console_get_key () != 0xff) {
         r = 1;
         break;
      }
      console_display_unblank ();
   }
   return r;
}
 
void prog_exit () {
   console_cls ();
}

// error messages
const char *msg_illegal_dimensions = "life: bad dimensions\n";
const char *msg_illegal_seed       = "life: bad seed\n";

// the main method
int main (int argc, char *argv[]) {
 
   int opt;
   long sl;
   int r = 0;

   // use the current screen size as the default universe size
   const int w_max = console_display_get_columns ();
   const int h_max = console_display_get_rows ();  
   w = w_max;
   h = h_max;

   // set the default number of games and cycles
   games  = 2000000000;
   cycles = 256;

   // process all options
   while ((opt = getopt (argc, argv, "c:g:h:w:s:")) != -1) {
      switch (opt) {
         case 'c':
            cycles = atoi (optarg);
            if (cycles < 1) {
               r = 1;
            }
            break;
         case 'g':
            games = atol (optarg);
            if (games < 1) {
               r = 1;
            }
            break;
         case 'h':                                                  // height
            h = atoi (optarg);
            if (h < 1 || h > h_max) {
               dylib.fputs (msg_illegal_dimensions, stderr);
               r = 1;
            }
            break;
         case 'w':                                                  // width
            w = atoi (optarg);
            if (w < 1 || w > w_max) {   
               dylib.fputs (msg_illegal_dimensions, stderr);
               r = 1;
            }
            break;
         case 's':                                                  // random seed
            sl = atol (optarg);
            if (sl > (long) RAND_MAX) {
               dylib.fputs (msg_illegal_seed, stderr);
               r = 1;
            } else {
               s = sl;
            }
            break;
         case '?':                                                  // unhandled option
         default:
            r = 1;
            break;
      }
   }

   argc -= optind;                                                  // skip pass processed arguments
   argv += optind;

   if (r) {                                                         // test for startup success
      dylib.fputs                                                   // print usage
         ("usage: life [-c cycles] [-g games] [-h height] [-w width] [-s random_seed]\n", 
          stdout);
   } else {       
      dylib.atexit (prog_exit);                                                                         
      srand (s);                                                    // set the random seed
      for (long g = 0; g < games; g++) {
         if (game_play (cycles)) {                                  // run the game. if return is non-zero a key was pressed to exit
            break;
         }
      }
   }
   
   return r;                                                        // return status
}
