/* lp.c
   This program prints files to the printer parallel port. If no
   file is specified, stdin is assumed.
   Usage: lp [file [... filen]]
   change log:
   12/26/2024 initial version
   01/20/2025 moved fputs to dylib
   01/20/2025 modified to not print a request message when no files were available for print
   05/03/2025 major updates including auto-determination of printer device path
   06/08/2025 removed redundant form feeds
   06/16/2025 updated exit reference
   12/16/2025 updated for printer spec
*/ 

#include <dylib.h>
#include <util.h>
#include <stdio.h>
#include <printer_private.h>
#include <stdlib.h>
#include <unistd.h>

// prints usage info
const char *usage = "usage: lp [-d printer] [file ...]\n";
void print_usage () {
   dylib.fputs (usage, stderr);
}

int main (int argc, char *argv[]) {

   FILE *fin;
   FILE *fout;

   char printer[FILENAME_MAX];
   char printer_dev_file[FILENAME_MAX];

   int count = 0;
   int have_files = 0;

   int opt;

   dylib.strcpy (printer, "lp");

   // capture the options
   while ((opt = getopt (argc, argv, "d:")) != -1) {
      switch (opt) {
         case 'd':                                               // handle the default color
            dylib.strcpy (printer, optarg);
            break;
         case '?':                                               // handle unknown options as errors
         default:
            print_usage ();
            exit (1);
            break;
      }
   }

   // capture the default printer
   int dstat = printer_get_dev_file (printer, printer_dev_file);

   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   if (!dstat) {

      fout = dylib.fopen (printer_dev_file, "w");
      if (fout) {
         if (argc) {
            for (int i = 0; i < argc; i++) {
               fin = dylib.fopen (argv[i], "r");
               if (fin) {
                  fcopy (fin, fout);
                  dylib.fclose (fin);
                  count++;
               } else {
                  fprintf (stderr, "lp: %s: No such file or directory\n", argv[i]);
               }
            }
            have_files = 1;
         } else {
            fcopy (stdin, fout);
         }
         dylib.fclose (fout);

         if ((have_files && count) || !have_files) {
            fprintf (stdout, "request id is %s-0 (%d file(s))\n", printer, count);
         }
      } else {
         fprintf (stderr, "lp: printer %s device %s failed to open\n", printer, printer_dev_file);
      }
   } else {
      fprintf (stderr, "lp: printer %s is not available\n", printer);
   }

   return 0;
}
