/* ls.c
   This program lists directory contents.
   Usage: ls [-l] path [... pathn]
   Options:
   -l long listing, showing file types, size in bytes and blocks
   change log:
   06/23/2023 initial version
   07/26/2023 added hack to force output to 1 column wide when not printing
              to a console window.
   08/05/2023 modified to list for cwd if no other directories are specified
   08/11/2023 added display of directory name when printing multiple directories
   08/17/2023 changed file_entry_t type and reclen fields to integers, and sectors to long. That reduced the number
              of malloc alloc units by 50%
   08/24/2023 updated long format output to better match the standard
   08/25/2023 updated to show directories as executable on long listing
   09/22/2023 updated error message when a directory cannot be opened
   11/16/2023 corrected line wrap
   11/21/2023 updated to use FILENAME_MAX
   12/10/2023 changed references to math.h
              removed reference to string_ext.h
   02/27/2024 updated to use dylib
   03/07/2024 updated to call dylib's getcwd
   01/20/2025 updated fputs to dylib
   05/31/2025 added support for listing files
   06/01/2025 added sorting of parameters and then sorting of directory contents
   06/04/2025 corrected usage statement
   06/11/2025 updated to display only directories as executable in long format
   06/16/2025 updated exit reference
   07/22/2025 updated usage
   09/16/2025 updated to limit the maximum number of files to process to 128 or until a record cannot be malloc'd
   10/04/2025 corrected call for dylib
*/

#include <string.h>
#include <stdio.h>
#include <statfs.h>
#include <dirent.h>
#include <stdlib.h>
#include <math.h>
#include <sys/ioctl.h>
#include <conversion.h>
#include <unistd.h>
#include <constants.h>
#include <dylib.h>
#include <sys/stat.h>

// file entry type
typedef struct {
   char name[FILENAME_MAX];   // 32
   int ftype;                 //  2
   long sectors;              //  4
   int reclen;                //  2
   bool protected;            //  2
} file_entry_t;               // 42 bytes total. When used with malloc the size should be this + 4 mgt bytes = 44

#define FILES_MAX_COUNT 128

// files type
typedef struct {
   file_entry_t *file[FILES_MAX_COUNT];
   int count;
} files_t;

// sorts file entries by name
int qsort_compare (const void *a, const void *b) {

    file_entry_t *a_entry = *(file_entry_t **)a;
    file_entry_t *b_entry = *(file_entry_t **)b;

    return dylib.strcmp (a_entry->name, b_entry->name);
}

// print a list of files
void files_print_list (files_t *files) {

   int collen = 0;
   int rows, cols;

   struct winsize w;
   int ir = ioctl (fileno (stdout), TIOCGWINSZ, &w);  // get the size of the screen window

   for (int i = 0; i < files->count; i++) {
      collen = max (dylib.strlen (files->file[i]->name), collen);
   }
   if (ir == UNDEFINED) {
      w.ws_row = 0;
      w.ws_col = collen + 1;                          // hack to force display to single column output
                                                      // when not writing to a console window.
   } else {
      w.ws_col--;                                     // reduce width by one to prevent line wrap
   }
   collen++;

   cols = w.ws_col / collen;
   rows = files->count / cols;     
   if (files->count % cols) rows++;

   char fmts[8];
   sprintf (fmts, "%%-%ds", collen);

   int v;
   for (int r = 0; r < rows; r++) {
      for (int c = 0; c < cols; c++) {
          v = r + c * rows;
          if (v < files->count) {
             fprintf (stdout, fmts, files->file[v]->name);
          }
      }
      dylib.fputs ("\n", stdout);
   }
}

// prints a list of files with attributes
void files_print_list_with_attr (files_t *files) {

   char dperm[12];

   for (int i = 0; i < files->count; i++) {

      // prepare the directory and permissions info
      dylib.strcpy (dperm, "----------");
      if (files->file[i]->ftype == 6) {
         dperm[0] = 'd';
      }
      if (files->file[i]->protected != 1) {
         dperm[1] = 'r';
         dperm[4] = 'r';
         dperm[7] = 'r';
         dperm[2] = 'w';
         dperm[5] = 'w';
         dperm[8] = 'w';
      }
      if (files->file[i]->ftype == 6) { // directories
         dperm[3] = 'x';
         dperm[6] = 'x';
         dperm[9] = 'x';
      }
      // output a single file and its data
      fprintf (stdout, "%s 1 x x %7ld %s\n", dperm, files->file[i]->sectors * (long) 256, files->file[i]->name);
   }
}

bool file_add_report = true;  // flag to output a max directory size exceeded error only once

// adds an entry to a files list
void files_add_entry (files_t *files, const char *name, int ftype, long sectors, int reclen, bool protected) {
   bool file_added = false;
   if (files->count < FILES_MAX_COUNT) {                           // prevent processing more than current memory allocations
      files->file[files->count] = malloc (sizeof (file_entry_t));  // allocate memory for this file entry
      if (files->file[files->count]) {                             // only add this entry if the memory was allocated successfully
         dylib.strcpy (files->file[files->count]->name, name);     // set all the values
         files->file[files->count]->ftype     = ftype;
         files->file[files->count]->sectors   = sectors;
         files->file[files->count]->reclen    = reclen;
         files->file[files->count]->protected = protected;
         files->count++;                                           // increment the count
         file_added = true;
      }
   }
   if (!file_added) {
      if (file_add_report) {
         dylib.fputs ("ls: max files in dir exceeded\n", stderr);
         file_add_report = false;
      }
   }
}

// frees the entries from a files list
void files_free (files_t *files) {
   for (int i = 0; i < files->count; i++) {
      free (files->file[i]);
   }
   files->count = 0;
}

// initializes a file list
void files_init (files_t *files) {
   files->count = 0;
}

// lists a directory
void dir_list (const char *path, bool list_attributes, bool print_dir_name) {
   struct dirent *e;
   DIR *d = opendir (path);
   files_t files;
   int r;
 
   if (d) {
      files_init (&files); 
      while ((e = readdir (d))) { 
                                                                  // a simple (long) typecast should work but there's a bug in
                                                                  // the C compiler, so call a conversion explicitly. even calling
                                                                  // __fixdfdi (double d) doesn't work. Woof!
         files_add_entry (&files, e->d_name, (int) e->d_type, ftol (e->d_sectors), (int) e->d_reclen, e->d_protected);
      }
      r = closedir (d);
      if (r) {
         dylib.fputs ("couldn't close the directory\n", stderr);
      }

      // print directory name if required
      if (print_dir_name) {
         // write the directory name
         dylib.fputs (path, stdout);
         dylib.fputs (":\n", stdout);
      }

      // sort the directory contents
      qsort (files.file, files.count, sizeof (file_entry_t *), qsort_compare);

      if (list_attributes) {
         files_print_list_with_attr (&files);
      } else {
         files_print_list (&files);
      }

      files_free (&files);
   }
} 

// lists all specified directories
void dirs_list (files_t *dirs, bool list_attributes, bool print_dir_name, bool space_skip) {
   for (int i = 0; i < dirs->count; i++) {
      if (space_skip) {
         dylib.fputs ("\n", stdout);
      } else {
         space_skip = true;
      }
      dir_list (dirs->file[i]->name, list_attributes, print_dir_name);
   }
}

// lists all files specified in the arg list
void files_list_all_args (int argc, char *argv[], bool list_attributes, bool print_dir_name) {

   files_t dirs;
   files_t files;
   int i, n;
   struct stat statbuf;

   files_init (&dirs);
   files_init (&files);

   bool space_skip = false;

   for (i = 0; i < argc; i++) {
      n = stat (argv[i], &statbuf);
      if (!n) {
         if (S_ISDIR (statbuf.st_mode)) {
            files_add_entry (&dirs, argv[i], (int) statbuf.st_mode, statbuf.st_size >> 8, 0, 0);
         } else {
            files_add_entry (&files, argv[i], (int) statbuf.st_mode, statbuf.st_size >> 8, 0, 0);
            space_skip = true;
         }
      } else {
         dylib.fputs ("ls: ", stderr);
         dylib.fputs (argv[i], stderr);
         dylib.fputs (": No such file or directory\n", stderr);
      }
   }

   // sort the file and directory parameters
   qsort (files.file, files.count, sizeof (file_entry_t *), qsort_compare);
   qsort (dirs.file, dirs.count, sizeof (file_entry_t *), qsort_compare);

   if (list_attributes) {
      files_print_list_with_attr (&files);
   } else {
      files_print_list (&files);
   }
   files_free (&files);

   dirs_list (&dirs, list_attributes, print_dir_name, space_skip);
   files_free (&dirs);
}

// the main method
int main (int argc, char *argv[]) {

   // capture the options
   int opt;
   bool error        = false;
   bool long_listing = false;
   while ((opt = getopt (argc, argv, "l")) != -1) {
      switch (opt) {
      case 'l':
         long_listing = true; 
         break;
      case '?':
         error = true;
         break;
      default:
         break; 
      } 
   }

   if (error) {
      dylib.fputs ("usage: ls [-l] [file ...]\n", stderr);
      exit (0);
   }

   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   // determine if listing is for specific directories or the cwd
   if (argc > 0) {
      files_list_all_args (argc, argv, long_listing, argc > 1);
   } else {
      char cwd[FILENAME_MAX];
      char *p = dylib.getcwd (cwd, sizeof (cwd));     // get the cwd
      if (p) {
         dir_list (cwd, long_listing, false);       // list the cwd
      } else {
         dylib.fputs ("ls: cwd not set\n", stdout);
      }
   }

   return 0;
}
