/* missile.c
   This program demonstrates the bitmap mode in its fully glory, using graphics, sprites and sound.
   change log:
   01/11/2025 initial version
   01/12/2025 updated graphics
              added scores and missile counts
              added rudimentary missile flight and removal behavior
              added aim point
   01/13/2025 updated line drawing
   02/28/2025 updated for removal of 80x30 display mode
   06/05/2025 signal handling
   06/07/2025 moved calls to dylib
*/

#include <stdio.h>
#include <console.h>
#include <conversion.h>
#include <string.h>
#include <dylib.h>
#include <stdlib.h>

#define KEY_UP    'e'
#define KEY_DOWN  'x'
#define KEY_LEFT  's'
#define KEY_RIGHT 'd'

#define MISSILE_STATE_INCOMING     2
#define MISSILE_STATE_REMOVE_TRAIL 1
#define MISSILE_STATE_DEAD         0

typedef struct {
   int y_org;
   int x_org;
   int y;
   int x;
   int y_vel;
   int x_vel;
   int x_index;
   int x_div;
   int y_term;
   int state;
   int sprite_id;
} missile_t;

void init_missile (missile_t *missile) {
   missile->state = MISSILE_STATE_DEAD;
}

void add_missile (missile_t *missile, int sprite_id, int y, int x, int y_term, int y_vel, int x_vel, int x_div) {
   missile->y_org     = y;
   missile->x_org     = x;
   missile->y         = y;
   missile->x         = x;
   missile->y_vel     = y_vel;
   missile->x_vel     = x_vel;
   missile->x_index   = 1;
   missile->x_div     = x_div;
   missile->y_term    = y_term;
   missile->state     = MISSILE_STATE_INCOMING;
   missile->sprite_id = sprite_id;
}

void update_missile (missile_t *missile) {
   if (missile->state) {
      missile->y += missile->y_vel;
      missile->x_index--;
      if (!missile->x_index) {
         missile->x += missile->x_vel;
         missile->x_index = missile->x_div;
      }
   }
}

void remove_missile (missile_t *missile) {
   int y, x;
   y                = missile->y;
   x                = missile->x;
   missile->y       = missile->y_org;
   missile->x       = missile->x_org;
   missile->x_index = 1;
   console_sprite_set_values (missile->sprite_id, 1, COLOR_LTRED, 192, 0, 0, 0);
   while (1) {
      console_bitmap_draw_pixel (missile->y, missile->x, CONSOLE_BITMAP_CLEAR);
      if (y == missile->y && x == missile->x) {
         break;
      }
      update_missile (missile);
   }
   missile->state = MISSILE_STATE_DEAD;
}

void draw_missile (missile_t *missile) {
   if (missile->state) {
      console_bitmap_draw_pixel (missile->y, missile->x, CONSOLE_BITMAP_SET);
      console_sprite_set_values (missile->sprite_id, 1, COLOR_LTRED, missile->y, missile->x, 0, 0);
// } else {
//    console_sprite_set_values (missile->sprite_id, 1, COLOR_LTRED, 192, 0, 0, 0);
   }
}

void perform_coincidence (missile_t * missile) {
   if (missile->state == MISSILE_STATE_INCOMING) {
      if (missile->y == missile->y_term) {
         missile->state = MISSILE_STATE_REMOVE_TRAIL;
      }
   }
}

void init_screen () {
   console_display_set_mode (DISPLAY_MODE_BITMAP, DISPLAY_ROWS_24);

   // clear the screen
   console_bitmap_cls (CONSOLE_BITMAP_IMAGE_TABLE_FILL_INCREMENTING);

   // set the patterns
   unsigned char char_pattern[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
   console_bitmap_set_all_patterns (char_pattern);

   // set the colors
   unsigned char color_pattern[] = {0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0};
   console_bitmap_set_all_colors (color_pattern);
}

void write_score (int y, int x, int v) {
   char s[] = {0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00};
   char *p = dylib.int2str (v);
   int len = dylib.strlen (p);;
   dylib.strcpy (&s[5 - len], p);
   s[5] = '0';
   console_bitmap_write_text_raw (y, x, s);
}

void write_missile_count (int y, int x, int v) {
   char s[] = {0x20, 0x20, 0x00, 0x00};
   char *p = dylib.int2str (v);
   int len = dylib.strlen (p);;
   dylib.strcpy (&s[2 - len], p);
   console_bitmap_write_text_raw (y, x, s);
}

const unsigned char missile_center_bitmap[] = {
   0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, 0xff
};

const unsigned char missile_center_color[] = {
   0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0,
   0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0,
   0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0,
   0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0
};

const unsigned char missile_bitmap[] = {
   0x01, 0x01, 0x01, 0x02, 0x00, 0x7f, 0x80, 0x00,
   0x00, 0x00, 0x00, 0x80, 0x00, 0xfe, 0x01, 0x00,
};

const unsigned char missile_color[] = {
   0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b,
   0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b
};

const unsigned char missile_count_color[] = {
   0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b,
   0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b, 0x4b
};

void draw_missile_center (int y, int x) {
   // draw missile center
   console_bitmap_set_pattern2 (y, x, missile_center_bitmap, sizeof (missile_center_bitmap));
   console_bitmap_set_color_pattern2 (y, x, missile_center_color, sizeof (missile_center_color));
  
   // draw missile
   console_bitmap_set_pixel_and_color ((y << 3) - 1, ((x + 2) << 3) - 1, CONSOLE_BITMAP_SET, 0x40);
   console_bitmap_set_pattern2 (y, x + 1, missile_bitmap, sizeof (missile_bitmap));
   console_bitmap_set_color_pattern2 (y, x + 1, missile_color, sizeof (missile_color));

   // set missile count color
   console_bitmap_set_color_pattern2 (y + 1, x + 1, missile_count_color, sizeof (missile_count_color));
}

const unsigned char city_bitmap[] = {
   0x18, 0x18, 0x19, 0x59, 0x5d, 0x5d, 0xfd, 0xff,
   0x40, 0x64, 0x64, 0x64, 0x6c, 0xfd, 0xff, 0xff
};

const unsigned char city_color[] = {
   0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40,
   0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40
};

void draw_city (int y, int x) {
   console_bitmap_set_pattern2 (y, x, city_bitmap, sizeof (city_bitmap));
   console_bitmap_set_color_pattern2 (y, x, city_color, sizeof (city_color));
}

const unsigned char ground_bitmap[] = {
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
};

const unsigned char ground_color[] = {
   0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0, 0xb0
};

void draw_ground () {
   for (int i = 0; i < 32; i++) {
      console_bitmap_set_pattern2 (23, i, ground_bitmap, sizeof (ground_bitmap));
      console_bitmap_set_color_pattern2 (23, i, ground_color, sizeof (ground_color));
   }
}

const unsigned char score_color[] = {
   0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40
};

void draw_score_area () {
   for (int i = 0; i < 32; i++) {
      console_bitmap_set_color_pattern2 (0, i, score_color, sizeof (score_color));
   }
}

void draw_game_field () {
   init_screen ();

   draw_ground ();

   draw_missile_center (22, 0);
   write_missile_count (23, 1, 30);
   draw_missile_center (22, 14);
   write_missile_count (23, 15, 30);
   draw_missile_center (22, 28);
   write_missile_count (23, 29, 30);

   draw_city (22, 5);
   draw_city (22, 8);
   draw_city (22, 11);
   draw_city (22, 19);
   draw_city (22, 22);
   draw_city (22, 25);

   draw_score_area ();
   write_score (0, 0, 32767);
   write_score (0, 13, 32767);
   write_score (0, 26, 32767);

   const unsigned char missile_pattern[] = {0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
   console_sprite_set_pattern (1, missile_pattern);

   int y = 92;
   int x = 124;
   const unsigned char aim_pattern[] = {0x10, 0x10, 0x10, 0xfe, 0x10, 0x10, 0x10, 0x00};
   console_sprite_set_pattern (0, aim_pattern);
   console_sprite_set_values (0, 0, 15, y, x, 0, 0);


   int key;
   int cont = 1;
   int i;
   int joy_y, joy_x;

   int add_missile_delay = 50;

   missile_t missile[3];
   for (i = 0; i < 3; i++) {
      init_missile (&missile[i]);
   }
// add_missile (&missile[0], 1, 8, 0, 184, 1, 1, 3);
// add_missile (&missile[1], 2, 8, 255, 184, 1, -1, 3);
// add_missile (&missile[2], 3, 8, 127, 184, 1, -1, 3);

   while (cont) {

      // check keyboard
      key = console_get_key ();

      // check for joystick and map to keys
      console_joystick_read (0, &joy_y, &joy_x);
      if (joy_y) {
         if (joy_y == JOYSTICK_UP) {
            key = KEY_UP;
         } else {
            key = KEY_DOWN;
         }
      } else if (joy_x) {
         if (joy_x == JOYSTICK_LEFT) {
            key = KEY_LEFT;
         } else {
            key = KEY_RIGHT;
         }
      }

      switch (key) {
         case KEY_UP:
            y--;
            if (y < 8) {
               y = 8;
            }
            break;
         case KEY_DOWN:
            y++;
            if (y > 160) {
               y = 160;
            }
            break;
         case KEY_LEFT:
            x--;
            if (x < 0) {
               x = 0;
            }
            break;
         case KEY_RIGHT:
            x++;
            if (x > 249) {
               x = 249;
            }
            break;
         case '1':
            cont = 0;
            break;
         default:
            break;
      }
  
      console_sprite_set_values (0, 0, 15, y, x, 0, 0);

      for (i = 0; i < 3; i++) {
         draw_missile (&missile[i]);
      }

      for (i = 0; i < 3; i++) {
         perform_coincidence (&missile[i]);
      }

      for (i = 0; i < 3; i++) {
         if (missile[i].state == MISSILE_STATE_REMOVE_TRAIL) {
            remove_missile (&missile[i]);
         }
      }

      for (i = 0; i < 3; i++) {
         update_missile (&missile[i]);
      }

      for (i = 0; i < 3; i++) {
         add_missile_delay--;
         if (!add_missile_delay) {
            add_missile_delay = 50;
            if (missile[i].state == MISSILE_STATE_DEAD) {
               switch (i) {
                  case 0:
                     add_missile (&missile[0], 1, 8, 0, 184, 1, 1, 3);
                     break;
                  case 1:
                     add_missile (&missile[1], 2, 8, 255, 184, 1, -1, 3);
                     break;
                  case 2:
                     add_missile (&missile[2], 3, 8, 127, 184, 1, -1, 3);
                     break;
                  default:
                     break;
               }
            }
         }
      }
   }
}

void play_game (int num_players) {
   draw_game_field ();
}

int display_title () {
   init_screen ();
   console_bitmap_write_text_raw (11, 8, "Missile Command");
   console_bitmap_write_text_raw (20, 0, "Press");
   console_bitmap_write_text_raw (21, 0, "1 for one player");
   console_bitmap_write_text_raw (22, 0, "2 for two players");
   console_bitmap_write_text_raw (23, 0, "X to exit");

   int cont = 1;
   int key;
   int num_players = 0;

   while (cont) {
      key = console_get_key ();
      switch (key) {
         case '1':
            num_players = 1;
            cont        = 0;
            break;
         case '2':
            num_players = 2;
            cont        = 0;
            break;
         case 'X':
         case 'x':
            num_players = 0;
            cont        = 0;
            break;
         default:
            break;
      }
   }

   return num_players;
}

void game_run () {
   int num_players;

   while (1) {
      num_players = display_title ();
      if (!num_players) {
         break;
      }
      play_game (num_players);
   }
}

int recover_display_mode;
int recover_display_rows;

// restore settings before exiting
void prog_exit () {
   console_display_set_mode (recover_display_mode, recover_display_rows);
   console_cls (); 
   console_standard_set_default_color ();
   console_fonts_load_std ();
}

int main (int argc, char *argv[]) {

   // save the current display mode
   console_display_get_mode (&recover_display_mode, &recover_display_rows);
 
   // set atexit
   dylib.atexit (prog_exit);

   // execute the game
   game_run ();
    
   return 0;
}
