/* mv.c
   This program implements the move command.
   limitations
   - implements only the -i, -n and -v options (the most common, and even then rarely needed)
   change history
   05/24/2025 initial version
   05/25/2025 added differentiation of POSIX case 1 (single rename/move of file or directory) and case 2 (more than 2
              parameters and the last exists and is a directory.
              added test for case 2. note that the processing of case 1 and 2 is the same as long as the case 2 
              conditions are met.
              added test to prevent processing of identical paths for file to file and dir to dir (copy p on top of itself).
   05/27/2025 moved multiple methods to unistd_private
              added test for mounted paths (cannot move)
              migrated all commanding into a set of batch methods, to be separated from mv.c
   05/28/2025 implemented the required directory and file manipulation operations (getting close)
   06/02/2025 added parameter to the copy_with_remove call to support the needs of the cp.c method
   06/03/2025 updated as necessary for cp/rm verbosity since all of these use shared utility methods
   06/10/2025 moved temp file to /tmp
*/

#include <utility_private.h>
#include <batch_private.h>
#include <dylib.h>
#include <unistd.h>
#include <unistd_private.h>

// displays program usage
void display_usage () {
   dylib.fputs ("usage: mv [-i | -n] [-v] source target\n", stderr);
   dylib.fputs ("       mv [-i | -n] [-v] source ... directory\n", stderr);
}

#define BATCH_NAME "/tmp/mv_t"

// the main procedure
int main (int argc, char *argv[]) {

   bool interactive_overwrite = false;                        // option flags
   bool overwrite_files       = true;
   bool verbose               = false;
   bool debug                 = false;

   bool error                 = false;                        // indicates whether option processing resulted in an erro

   int opt;
   while ((opt = getopt (argc, argv, "invd")) != -1) {        // capture the options
      switch (opt) {
         case 'i':                                            // interactive overwrite
            overwrite_files       = false;
            interactive_overwrite = true;
            break;
         case 'n':                                            // prevent overwrite
            overwrite_files       = false;
            interactive_overwrite = false;
            break;
         case 'v':                                            // verbosity
            verbose               = true;
            break;
         case 'd':
            debug                 = true;
            break;
         case '?':                                            // unrecognized option
         default:                                             // error
            error = true;                      
            break;
      }
   }

   int r = 0;                                                 // initial return result
   int r2;

   if (!error) {                                              // continue executing if option processing succeeded

      argc -= optind;                                         // adjust argc and argv removing the processed options
      argv += optind;
   
      if (argc >= 2) {                                        // ensure there are at least two parameters
         if (argc == 2 || is_dir (argv[argc - 1])) {          // confirm POSIX case 1 or case 2
            batch_t batch = batch_create (BATCH_NAME, verbose, false, debug);
            for (int i = 0; i < argc - 1; i++) {              // process all the parameters
               if (!dsr_path_is_root (argv[i])) {             // ensure parameter isn't a mounted filesystem
                  r2 = copy_with_remove                       // perform the operation
                          ("mv",
                           true,
                           true,
                           &batch,
                           argv[i], 
                           argv[argc - 1], 
                           overwrite_files, 
                           interactive_overwrite, 
                           verbose);
                  r += r2;
               } else {                                       // write diagnostic message re mounted paths
                  dylib.fputs ("mv: ", stderr);
                  dylib.fputs (argv[i], stderr);
                  dylib.fputs (": cannot move mounted path\n", stderr);
                  r = 1;
               }
            }
            batch_close (&batch);
            batch_execute (&batch);
            batch_remove (&batch);
         } else {                                             // handle the POSIX option 2 case failed (last arg doesn't exist or isn't a directory
            dylib.fputs ("mv: ", stderr);
            dylib.fputs (argv[argc - 1], stderr);
            dylib.fputs (" is not a directory\n", stderr);
            r = 1;
         }
      } else {
         display_usage ();                                    // display usage
         r = 1;                                               // set to failed
      }

   } else {
      display_usage ();                                       // display usage since there was an error processing options
      r = 1;
   }

   return r;                                                  // return the result
}
