/* pacman.c
   This program implements the lovable 1980's arcade game PacMan. It's intended to be a reasonable facsimile that retains
   much of the original fun. The monster patterns are entirely unfaithful to that of the original game as they are far more
   aggressive and effective.
   Cheats:
   - while in play, press 9 to disable powerpellets from being removed after consumption, thus proving an unlimited supply
   change log:
   08/12/2023 initial version
   09/27/2023 migrated sprite initialization to the console package
   09/28/2023 added pacman chomping graphics
              corrected player start delay
              updated to hide entities while the player start delay is in process
              corrected player id handling where player 1 moved to 2 even in a one player game
   09/29/2023 added game start selection screen
              added game font period symbol
              added copyright symbol
              migrated private sequencer call to console_get_key()
              corrected drawing of scores on game start selection screen
              added logic to stop motion but wait for sound to complete after player death
              integrated monster movement from text-based pac-Dan game
              separated pacman and ghosts into separate types to eliminate unused fields for each entity type
              added ghost state (angry, scared, eaten)
   09/30/2023 added display of pacmen lives
   10/07/2023 added ghost personalities and patterns
              migrated font definitions to load file to reduce program image size
              added powerpellets with flashing sprites
              added making ghosts scared (with blue color, fleeing and timeout) when a power pellet is eaten
              added flying bonuses when eating ghosts
   10/08/2023 added cooridor teleporting for pacman and ghosts
              corrected bug where player changing directions immediately at teleport locks the pacman outside of the grid
              added cheat, pressing 9, that eaten power pellets aren't removed, thus resulting in an unlimited supply
              added treat generation, liveliness and processing
              corrected bug where the wave can transition when the power pellets haven't all been eaten
   10/15/2023 corrected player ready call to better match original game
              added flashing player id while playing game
   10/21/2023 added writing of game over
              removed direct references to vdp.h replacing with console
   10/22/2023 added game grid flash after clearing all pellets
   10/23/2023 added sequencer actions for flashing the power pellets and the player 1UP/2UP identifers
              modified all delay loops to allow cycling of the sequencer
              moved sprite indexes for the ghost eaten bonus to occur immediately after the four ghosts
   10/25/2023 added sequencer throttling
              added bonus display for eating treats
              added bonus pacman at threshold score
              adjusted timing of delays at startup, restart after killed, end of game, etc.
              corrected treat index adding to only occur if one was eaten
   10/26/2023 modified ghost movement to move faster at grid alignment when ghosts are angry/eaten. This provides a 12.5%
              speed advantage over the pacman except after power pellets have been eaten
              modified pacman movement to move faster at grid alignment for a period after a power pellet has been eaten. This
              provides a 12.5% speed advantage over the ghosts that haven't yet been eaten, effectivly turning the tables on them.
              added scared ghost flashing when power pellet cycle is ending
              migrated the bonus display to the sequencer, correcting issue where bonuses didn't get hidden after pacman death
              created treat graphics
   10/28/2023 refactoring to create common methods for previous duplicate code
   10/29/2023 updated ghost scared time so it is depending on the levels
              updated to prevent pacman from entering the ghost house through the door
   10/31/2023 removed temp char defs for treats from the fonts file 
              added fruits to the fonts file using char color ranges 27-31 which fit the number of colors for the fruits. This
              is very wasteful for char range resources but the game is almost complete so it fits. That's the pain of not having
              bitmap graphics.
              added display of fruits that have been eaten
              added ghosts scared sqiggly mouths
   11/03/2023 modified to locate fonts based on the pacman game path (argv[0]) in order to disuse private APIs
              added pacman death sequence
   11/04/2023 added ghost eaten bonus sound and delay
              added munching sound
   11/05/2023 modified to prevent redrawing the game select on every key press
   11/21/2023 updated to use FILENAME_MAX
   12/10/2023 changed references to math.h
              removed reference to string_ext.h
              added reference to conversion.h
   02/27/2024 added use of dylib
   03/02/2024 updated with further use of dylib
   03/11/2024 added use of joysticks
   05/18/2024 updated to use keys mapping to the arrows on the TI-99 keyboard
              added macro definitions for the keys
              added toggling on/off power-pellets go away after being eaten
   05/19/2024 updated displayed copyright date
   12/27/2024 updated to restore the display mode
              updated to clear the display and print pacman name in the center of the screen
   01/20/2025 moved fputs to dylib
   01/20/2025 moved console_write_raw to dylib
              added kludge in displaying game over to what seems to be a compiler problem with constant strings. 
   01/22/2025 performance improvements
   02/28/2025 updated for removal of 80x30 mode
   05/30/2025 added eyes
              corrected timeout for increased speed to match ghost timeout after eating power pelleet
   06/05/2025 updated to correctly handle signals
   06/07/2025 moved calls to dylib
   06/12/2025 adjusted coincidence sensitivity from 7 to 5
   07/05/2025 dylib adjustments
   10/24/2025 forced background colors to black
   10/25/2025 added speed setting

   to do:
   - slow down ghosts going through the side teleport tunnel
   - fix ghost sequencing timing, so they come out, go to their haunt locations and then attack, and eaten sequences
   - add sounds extra play, Pac-Man eaten, coin inserted
   - fix ghost standard sequences. When scared the sequence should move to getting away from the Pac-Man 
   - determine how to slow things down more
   - add reward cartoons every three levels
   - add attract mode
   - figure out how to use vsync. will clean up the ghost eyes that occasionally are misaligned by one pixel
   - cleanup / refactoring
*/

#include <vdp.h>
#include <constants.h>
#include <console.h>
#include <stdio.h>
#include <stdlib.h>
#include <soundqueue.h>
#include <string.h>
#include <sequencer.h>
#include <math.h>
#include <libgen.h>
#include <conversion.h>
#include <dylib.h>
#include <signal.h>
#include <unistd.h>

// the template playing field
const char *template_play_field[] = {
   "      1UP    HIGH     2UP  \n",
   "        00      00      00 \n",
   "    abbbbbbbbbbdbbbbbbbbbbc\n",
   "    e__________e__________e\n",
   "    e^fbg_fbbg_h_fbbg_fbg^e\n", 
   "    e_____________________e\n",
   "    e_fbg_i_fbbjbbg_i_fbg_e\n",
   "    e_____e____e____e_____e\n",
   "    kbbbc_lbbg h fbbm_abbbn\n",   
   "        e_e         e_e    \n",   
   "    bbbbn_h ab---bc h_kbbbb\n",  
   "    $    _  e     e  _    #\n",   
   "    bbbbc_i kbbbbbn i_abbbb\n",   
   "        e_e         e_e    \n",   
   "    abbbn_h fbbjbbg h_kbbbc\n",   
   "    e__________e__________e\n",
   "    e_fbc_fbbg_h_fbbg_abg_e\n", 
   "    e^__e______ ______e__^e\n",
   "    lbg_h_i_fbbjbbg_i_h_fbm\n",
   "    e_____e____e____e_____e\n",
   "    e_fbbbobbg_h_fbbobbbg_e\n",
   "    e_____________________e\n",
   "    kbbbbbbbbbbbbbbbbbbbbbn\n",
   "                           \n"};

#define NUM_GHOSTS       4
#define NUM_POWERPELLETS 4

#define GHOST_STATE_ANGRY  0
#define GHOST_STATE_SCARED 1
#define GHOST_STATE_EATEN  2

#define GHOST_EXTRA_PIXEL_MULTIPLER   2
#define PACMAN_EXTRA_PIXEL_MULTIPLIER 2

#define BONUS_200  0
#define BONUS_400  1
#define BONUS_800  2
#define BONUS_1600 3

#define GHOST_HOLD_MOVE_SEQ_FIRST           0
#define GHOST_HOLD_MOVE_SEQ_LAST            (GHOST_HOLD_MOVE_SEQ_FIRST           +  24)
#define GHOST_HOUSE_EXIT_MOVE_SEQ_FIRST     (GHOST_HOLD_MOVE_SEQ_LAST            +   1)
#define GHOST_HOUSE_EXIT_MOVE_SEQ_LAST      (GHOST_HOUSE_EXIT_MOVE_SEQ_FIRST     +   5)
#define GHOST_OWN_AREA_SEQ_FIRST            (GHOST_HOUSE_EXIT_MOVE_SEQ_LAST      +   1)
#define GHOST_OWN_AREA_SEQ_LAST             (GHOST_OWN_AREA_SEQ_FIRST            +  97)
#define GHOST_CHASE_SEQ_FIRST               (GHOST_OWN_AREA_SEQ_LAST             +   1)
#define GHOST_CHASE_SEQ_LAST                (GHOST_CHASE_SEQ_FIRST               + 127)
#define GHOST_RESET_MOVE_SEQ                (GHOST_CHASE_SEQ_LAST                +   1)
#define GHOST_RETURN_TO_HOME_MOVE_SEQ_FIRST (GHOST_RESET_MOVE_SEQ                +   1)
#define GHOST_RETURN_TO_HOME_MOVE_SEQ_LAST  (GHOST_RETURN_TO_HOME_MOVE_SEQ_FIRST +  64)

#define NUM_TREAT_TYPES  16
#define TREAT_POS_Y      13
#define TREAT_POS_X      15

// pacman info type
typedef struct {
   int sprite_index;
   int dir;
   int image_index;
   int state_timeout;
   bool remove_extra_pixel_move;
} pacman_t;

// ghost info type
typedef struct {
   int sprite_index[3];
   int color;
   int state;
   int state_timeout;
   int move_seq_id;
   bool remove_extra_pixel_move;
} ghost_t;

// bonus info type
typedef struct {
   int sprite_index[2];
   int value_index;
   int state_timeout;
} bonus_t;

// treat info type
typedef struct {
   int treat_index;
   int state_seq_id;
} treat_t;

// player info type
typedef struct {
   int score;
   int extra_life_score;
   char play_field[24][32];
   int pellets_remaining;
   int lives_remaining;
   bool is_start_of_game;
   treat_t treat;
   int level;
} player_t;

// the game info type
typedef struct {
   int player_id;
   player_t player[2];
   bool player_is_alive;
   int num_players;
   int high_score;
   pacman_t pacman;
   ghost_t ghost[NUM_GHOSTS];
   bonus_t bonus;
   int speed;
} game_t;

// the game info
game_t game;

#define PLAYFIELD_BORDER_CHAR_FIRST    'a'
#define PLAYFIELD_BORDER_CHAR_LAST     'o'
#define PLAYFIELD_BORDER_PEN_DOOR_CHAR '-'

// keyboard directions
#define KEY_UP    'e'
#define KEY_DOWN  'x'
#define KEY_LEFT  's'
#define KEY_RIGHT 'd'

// ghost character ids by state and display sequence (only 2)
const int ghost_char_id[3][2] = {
   {112, 113},
   {114, 115},
   {112, 113}
};

// ghost eye characters based on direction [y][x], offset by one, since the direction is +/- 1, +/- 1
const int ghost_eye_char_id[3][3] = {
   {  0, 118,   0},  // X    up    X
   {116,   0, 117},  // left X     right
   {  0, 119,   0}   // X    down  X
};

// standard ghost colors
const int ghost_color[4][3] = {
   {COLOR_CYAN,    COLOR_DKBLUE, COLOR_BLACK},
   {COLOR_DKRED,   COLOR_DKBLUE, COLOR_BLACK},
   {COLOR_LTRED,   COLOR_DKBLUE, COLOR_BLACK},
   {COLOR_MAGENTA, COLOR_DKBLUE, COLOR_BLACK}
};

// pacman movement character ids
const int pacman_move_char_id[4][8] = { 
   {128, 129, 130, 131, 132, 131, 130, 129},
   {128, 137, 138, 139, 140, 139, 138, 137},
   {128, 133, 134, 135, 136, 135, 134, 133},
   {128, 141, 142, 143, 144, 143, 142, 141}
};

// pacman eaten character ids
#define PACMAN_DEATH_NUM_SEQ 7
const int pacman_death_char_id[PACMAN_DEATH_NUM_SEQ] = {184, 185, 186, 187, 188, 189, 190};

// ghost eaten bonus characters
const int ghost_bonus_char_id[4][2] = {
   {145, 149},
   {146, 149},
   {147, 149},
   {148, 149}
};

// ghost eaten bonus values
const int ghost_bonus_val[4] = {20, 40, 80, 160};

// ghost scared time per level
const int ghost_scared_time_for_level[16] = {80, 65, 50, 40, 80, 35, 30, 25, 20, 20, 80, 20, 20, 20, 20, 20};

// treat bonus characters
const int treat_bonus_char_id[16][2] = {
   {166, 149},  // 100
   {145, 149},  // 200
   {167, 149},  // 300
   {146, 149},  // 400
   {168, 149},  // 500
   {169, 149},  // 600
   {170, 149},  // 700
   {147, 149},  // 800
   {171, 149},  // 900
   {172, 149},  // 1000
   {173, 149},  // 1100
   {174, 149},  // 1200
   {175, 149},  // 1300
   {176, 149},  // 1400
   {177, 149},  // 1500
   {148, 149}   // 1600
};

// treat character ids
const unsigned char treat_char_id[16]   = {216, 217, 224, 225, 232, 240, 241, 248, 248, 248, 248, 248, 248, 248, 248, 248};

// treat colors
const unsigned char treat_char_color[5] = {COLOR_DKRED, COLOR_LTRED, COLOR_LTGREEN, COLOR_DKYELLOW, COLOR_LTBLUE};

// the treat bonus values
const int treat_bonus_val[16] = {10, 20, 30, 40, 50, 60, 70, 80, 90, 100, 110, 120, 130, 140, 150, 160};

// locations of ghosts hold points:

// the ghost's house
const int ghost_house_hold_center_y[NUM_GHOSTS] = {17 * 8 - 1, 17 * 8 - 1, 17 * 8 - 1, 17 * 8 - 1};
const int ghost_house_hold_center_x[NUM_GHOSTS] = {15 * 8,     15 * 8,     15 * 8,     15 * 8};

// the house exits (just outside the house)
const int ghost_house_exit_center_y[NUM_GHOSTS] = {9  * 8 - 1,  9 * 8 - 1, 9  * 8 - 1, 9  * 8 - 1 };
const int ghost_house_exit_center_x[NUM_GHOSTS] = {11 * 8,     19 * 8,     11 * 8,     19 * 8 };

// the ghosts' haunt areas
const int ghost_haunt_area_center_y[NUM_GHOSTS] = {4 * 8 - 1, 4  * 8 - 1, 17 * 8 - 1, 17 * 8 - 1};
const int ghost_haunt_area_center_x[NUM_GHOSTS] = {5 * 8,     25 * 8,      5 * 8,     25 * 8};

// returns the startup music
void sound_get_seq_startup_music (char **p, int *len) {
   static const char this_sound[] = {
      0x06, 0x89, 0x38, 0x90, 0xa2, 0x0e, 0xb0, 0x06, 
      0x04, 0x87, 0x09, 0x90, 0xbf, 0x06, 
      0x03, 0xa4, 0x1c, 0xb0, 0x06, 
      0x03, 0x83, 0x0b, 0x90, 0x06, 
      0x01, 0xbf, 0x06, 
      0x06, 0x89, 0x38, 0x90, 0xa2, 0x0e, 0xb0, 0x06, 
      0x04, 0x87, 0x09, 0x90, 0xbf, 0x06, 
      0x03, 0xa4, 0x1c, 0xb0, 0x06, 
      0x03, 0x83, 0x0b, 0x90, 0x06, 
      0x01, 0xbf, 0x06, 
      0x06, 0x8e, 0x6a, 0x90, 0xa5, 0x0d, 0xb0, 0x06, 
      0x04, 0x8e, 0x08, 0x90, 0xbf, 0x06, 
      0x03, 0xa7, 0x35, 0xb0, 0x06, 
      0x03, 0x89, 0x0a, 0x90, 0x06, 
      0x01, 0xbf, 0x06, 
      0x06, 0x8e, 0x6a, 0x90, 0xa5, 0x0d, 0xb0, 0x06, 
      0x04, 0x8e, 0x08, 0x90, 0xbf, 0x06, 
      0x03, 0xa7, 0x35, 0xb0, 0x06, 
      0x03, 0x89, 0x0a, 0x90, 0x06, 
      0x01, 0xbf, 0x06, 
      0x06, 0x89, 0x38, 0x90, 0xa2, 0x0e, 0xb0, 0x06, 
      0x04, 0x87, 0x09, 0x90, 0xbf, 0x06, 
      0x03, 0xa4, 0x1c, 0xb0, 0x06, 
      0x03, 0x83, 0x0b, 0x90, 0x06, 
      0x01, 0xbf, 0x06, 
      0x06, 0x89, 0x38, 0x90, 0xa2, 0x0e, 0xb0, 0x06, 
      0x04, 0x87, 0x09, 0x90, 0xbf, 0x06, 
      0x03, 0xa4, 0x1c, 0xb0, 0x06, 
      0x03, 0x83, 0x0b, 0x90, 0x06, 
      0x01, 0xbf, 0x06, 
      0x01, 0x9f, 0x06, 
      0x04, 0x90, 0xac, 0x25, 0xb0, 0x06, 
      0x04, 0x89, 0x0a, 0x90, 0xbf, 0x06, 
      0x03, 0x80, 0x0a, 0x90, 0x06, 
      0x06, 0x8a, 0x21, 0x90, 0xa7, 0x09, 0xb0, 0x06, 
      0x04, 0x8e, 0x08, 0x90, 0xbf, 0x06, 
      0x06, 0x86, 0x08, 0x90, 0xaf, 0x0e, 0xb0, 0x06, 
      0x04, 0x8f, 0x07, 0x90, 0xbf, 0x06, 
      0x03, 0x87, 0x07, 0x90, 0x06, 
      0x06, 0x82, 0x0e, 0x90, 0xa1, 0x07, 0xb0, 0x06, 
      0x02, 0x9f, 0xbf, 0x00
   };
   *p   = (char *) this_sound;
   *len = sizeof (this_sound);
};

// plays the startup music
int h_startup_music = UNDEFINED;
void __attribute__ ((noinline)) sound_play_startup_music () {
   if (h_startup_music == UNDEFINED) {
      char *p;
      int len;
      sound_get_seq_startup_music (&p, &len);
      h_startup_music = soundqueue_load (p, len, SOUNDQUEUE_PREEMPTION_RULE_ABORT);
   }
   soundqueue_play (h_startup_music);
}

// delays game play while sound is playing while continuing to advance the sequencer
void delay_until_sound_end () {
   sequencer_set_throttle (256);
   while (!soundqueue_play_is_done ()) {
      console_get_key ();
   }
   sequencer_set_throttle (0);
}

// initializes the character definitions from file
void chardefs_init (const char *argv0) {
   char fonts_path[FILENAME_MAX];
   dylib.strcpy (fonts_path, dirname ((char *)argv0));
   dylib.strcat (fonts_path, "/pacfonts");
   console_font_load (fonts_path);
}

// initializes the character colors
void colors_init () {
   // black background
   console_border_color_set (COLOR_BLACK);
   console_text_set_background_color (COLOR_BLACK);
   console_text_set_foreground_color (COLOR_WHITE);

   // regular text
   for (int i = 4; i < 12; i++) {
      console_standard_set_char_group_color (i, COLOR_WHITE, COLOR_TRANS);
   }

   // playfield border
   for (int i = 12; i < 16; i++) {
      console_standard_set_char_group_color (i, COLOR_DKBLUE, COLOR_TRANS);
   }

   // pacman defs
   for (int i = 16; i < 18; i++) {
      console_standard_set_char_group_color (i, COLOR_DKYELLOW, COLOR_TRANS);
   }

   // all else (unused)
   for (int i = 18; i < 26; i++) {
      console_standard_set_char_group_color (i, COLOR_GRAY, COLOR_TRANS);
   }

   int j = 0;
   for (int i = 27; i < 32; i++) {
      console_standard_set_char_group_color (i, treat_char_color[j], COLOR_TRANS);
      j++;
   }
}

// terminates the program
void prog_reset_display (int recover_display_mode, int recover_display_rows) {
   console_display_set_mode (recover_display_mode, recover_display_rows);
   console_cls ();
   console_standard_set_default_color ();
   console_fonts_load_std ();
}

// returns the gobble sound sequence
void sound_get_seq_gobble (char **p, int *len) {
   static const char pl_gobble[] = {
      0x06,       // bytes
      0x84, 0x36, // tone1 frequency, 116 Hz, xyz=364, order is zxy
      0x92,       // tone1 volume 2
      0xbf,       // tone2 volume off
      0xdf,       // tone3 volume off
      0xff,       // noise volume off
      0x01,       // duration

      0x03,       // bytes
      0x8d, 0x2c, // tone1 freqeuncy, 156 Hz, xyz=2cd, order is zxy
      0x90,       // tone1 volume max
      0x01,       // duration

      0x03,       // bytes
      0x89, 0x3f, // tone1 frequency, 110 Hz, xyz=3f9, order is zxy
      0x94,       // tone1 volume 4
      0x01,       // duration

      0x01,       // one byte for freq/vol info
      0x9f,       // tone1 volume off
      0x00        // duration
   };
   *p   = (char *) pl_gobble;
   *len = sizeof (pl_gobble);
}

// returns the pacman death sound sequence
void sound_get_seq_pacman_death (char **p, int *len) {
   static const char pl_pacman_death[] = {
      0x06,       // bytes
      0x84, 0x36, // tone1 frequency, 116 Hz, xyz=364, order is zxy
      0x92,       // tone1 volume 2
      0xbf,       // tone2 volume off
      0xdf,       // tone3 volume off
      0xff,       // noise volume off
      0x01,       // duration

      0x03,       // bytes
      0x8d, 0x2c, // tone1 freqeuncy, 156 Hz, xyz=2cd, order is zxy
      0x90,       // tone1 volume max
      0x01,       // duration

      0x03,       // bytes
      0x89, 0x3f, // tone1 frequency, 110 Hz, xyz=3f9, order is zxy
      0x94,       // tone1 volume 4
      0x01,       // duration

      0x01,       // one byte for freq/vol info
      0x9f,       // tone1 volume off
      0x00        // duration
   };
   *p   = (char *) pl_pacman_death;
   *len = sizeof (pl_pacman_death);
}

void sound_get_seq_ghost_eaten (char **p, int *len) {
   static const char pl_ghost_eaten[] = {
      0x06,       // bytes 
      0x88, 0x3f, // tone1 frequency
      0x92,       // tone1 volume 2
      0xbf,       // tone2 volume off
      0xdf,       // tone3 volume off
      0xff,       // noise volume off 
      0x02,       // duration
   
      0x03,       // bytes
      0x81, 0x2d, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x8f, 0x22, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x88, 0x1c, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x81, 0x18, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x8d, 0x14, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x86, 0x12, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x87, 0x10, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x8e, 0x0e, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x89, 0x0d, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x87, 0x0c, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x88, 0x0b, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x03,       // bytes
      0x8c, 0x0a, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration 
   
      0x01,       // one byte for freq/vol info
      0x9f,       // tone1 volume off
      0x06,       // duration

      0x03,       // bytes
      0x8c, 0x0a, // tone1 freqeuncy
      0x90,       // tone1 volume max
      0x02,       // duration

      0x03,       // bytes
      0x8c, 0x0a, // tone1 freqeuncy
      0x93,       // tone1 volume max
      0x03,       // duration

      0x01,       // one byte for freq/vol info
      0x9f,       // tone1 volume off
      0x00        // duration
   };
   *p   = (char *) pl_ghost_eaten;
   *len = sizeof (pl_ghost_eaten);
}

// plays the gobble sound sequence
void __attribute__ ((noinline)) sound_play_gobble () {
   static int h_gobble = UNDEFINED;
   if (h_gobble == UNDEFINED) {
      char *p;
      int len;
      sound_get_seq_gobble (&p, &len);
      h_gobble = soundqueue_load (p, len, SOUNDQUEUE_PREEMPTION_RULE_ABORT);
   }
   soundqueue_play (h_gobble);
}

// plays the gobble sound sequence
void __attribute__ ((noinline)) sound_play_ghost_eaten () {
   static int h_ghost_eaten = UNDEFINED;
   if (h_ghost_eaten == UNDEFINED) {
      char *p;
      int len;
      sound_get_seq_ghost_eaten (&p, &len);
      h_ghost_eaten = soundqueue_load (p, len, SOUNDQUEUE_PREEMPTION_RULE_ABORT);
   }
   soundqueue_play (h_ghost_eaten);
}

// plays the pacman_death sound sequence
void __attribute__ ((noinline)) sound_play_pacman_death () {
   static int h_pacman_death = UNDEFINED;
   if (h_pacman_death == UNDEFINED) {
      char *p;
      int len;
      sound_get_seq_pacman_death (&p, &len);
      h_pacman_death = soundqueue_load (p, len, SOUNDQUEUE_PREEMPTION_RULE_ABORT);
   }
   soundqueue_play (h_pacman_death);
}

// initializes the play field
void playfield_init (int player_id) {
   for (int i = 0; i < 24; i++) {
      dylib.strcpy (game.player[player_id].play_field[i], template_play_field[i]);
   }
}

// draws the play field
void playfield_draw (bool do_clear_screen) {
   if (do_clear_screen) {
      console_cls ();
   }
   int len = dylib.strlen (&game.player[game.player_id].play_field[0][0]) - 1;
   for (int i = 0; i < 24; i++) {
      console_write_raw (i, 0, &game.player[game.player_id].play_field[i][0], len);
   }
}

// hides all the ghosts
void ghosts_hide () {
   // hide the ghosts
   for (int i = 0; i < NUM_GHOSTS; i++) {          
      console_sprite_set_values (game.ghost[i].sprite_index[0], 0, 0, 192, 0, 0, 0);
      console_sprite_set_values (game.ghost[i].sprite_index[1], 0, 0, 192, 0, 0, 0);
      console_sprite_set_values (game.ghost[i].sprite_index[2], 0, 0, 192, 0, 0, 0);
   }
}

// hides all the ghosts and pacman
void entities_hide () {

   // hide the pacman
   console_sprite_set_values (game.pacman.sprite_index, 0, 0, 192, 0, 0, 0);

   // hides the ghosts
   ghosts_hide ();
}

// stops all the ghosts and pacman
void entities_stop () {
   console_sprite_set_motion (game.pacman.sprite_index, 0, 0);
   for (int i = 0; i < NUM_GHOSTS; i++) {
      console_sprite_set_motion (game.ghost[i].sprite_index[0], 0, 0);
      console_sprite_set_motion (game.ghost[i].sprite_index[1], 0, 0);
      console_sprite_set_motion (game.ghost[i].sprite_index[2], 0, 0);
   }
}

// power pellet char defs
const unsigned char powerpellet_present_char_def[8] = {0x00, 0x00, 0x18, 0x3C, 0x3C, 0x18, 0x00, 0x00};
const unsigned char powerpellet_blank_char_def[8]   = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

// blinks the power pellets
void powerpellets_blink_service () {
   static int cycle_id = 0;
   switch (cycle_id) {
      case 0:
         console_font_set (94, powerpellet_present_char_def);
         break;
      case 12:
         console_font_set (94, powerpellet_blank_char_def);
         break;
      case 24:
         cycle_id = -1;
         break;
      default:
         break;
   }
   cycle_id++;
}

// the player id flashing values
const char *oneup   = "1UP";
const char *twoup   = "2UP";
const char *clearup = "   ";

// blinks the player id
void playerid_blink_service () {
   static int cycle_id = 0;
   switch (cycle_id) {
      case 0: 
         console_write_raw (0, 6, oneup, 3);
         console_write_raw (0, 22, twoup, 3);
         break;
      case 12:
         if (game.player_id == 0 && game.player[0].lives_remaining) {
            console_write_raw (0, 6, clearup, 3); 
         }
         if (game.player_id == 1 && game.player[1].lives_remaining) {
            console_write_raw (0, 22, clearup, 3);
         }
         break;
      case 24:
         cycle_id = -1; 
         break;
      default:
         break;
   }
   cycle_id++;
}  

// displays bonuses
void bonus_display_service () {
   if (game.bonus.state_timeout) {
      game.bonus.state_timeout--;
      if (!game.bonus.state_timeout) {
         console_sprite_set_color (game.bonus.sprite_index[0], COLOR_TRANS);
         console_sprite_set_color (game.bonus.sprite_index[1], COLOR_TRANS);
         console_sprite_set_motion (game.bonus.sprite_index[0], 0, 0);
         console_sprite_set_motion (game.bonus.sprite_index[1], 0, 0);
      }
   }
}

// initializes bonus info
void bonus_init () {
   game.bonus.sprite_index[0] = 5;
   game.bonus.sprite_index[1] = 6;
}

// shows ghost eaten bonus
void ghost_bonus_show (int bonus_id, int y, int x) {
   console_sprite_set_values (game.bonus.sprite_index[0], ghost_bonus_char_id[bonus_id][0], COLOR_WHITE, y, x, 0, 0);
   console_sprite_set_values (game.bonus.sprite_index[1], ghost_bonus_char_id[bonus_id][1], COLOR_WHITE, y, x + 8, 0, 0);
   game.bonus.state_timeout = 80;
}

// shows treat eaten bonus
void treat_bonus_show (int bonus_id, int y, int x) {
   console_sprite_set_values (game.bonus.sprite_index[0], treat_bonus_char_id[bonus_id][0], COLOR_WHITE, y, x, 0, 0);
   console_sprite_set_values (game.bonus.sprite_index[1], treat_bonus_char_id[bonus_id][1], COLOR_WHITE, y, x + 8, 0, 0);
   game.bonus.state_timeout = 100; 
}   

// returns the correct color for the ghost and its state
int ghost_get_color (int id) {
   int r = ghost_color[id][game.ghost[id].state];
   if (game.ghost[id].state == GHOST_STATE_SCARED) {
      if (game.ghost[id].state_timeout < 20 && game.ghost[id].state_timeout % 2 == 0) {
         r = COLOR_WHITE;
      }
   }
   return r;
}

// initializes the ghosts and pacman values
void entities_init () {

   // initialize the pacman
   game.pacman.dir                     = 3; 
   game.pacman.sprite_index            = 0;
   game.pacman.state_timeout           = 0;    // power pellet mode
   game.pacman.remove_extra_pixel_move = false;

   console_sprite_set_values 
      (game.pacman.sprite_index, 128, COLOR_DKYELLOW, 7 + 16 * 8, 15 * 8, 0, 0);

   // initialize the ghosts
   for (int i = 0; i < NUM_GHOSTS; i++) {
      game.ghost[i].sprite_index[0] = i + 1;
      game.ghost[i].sprite_index[1] = i + 7;
      game.ghost[i].sprite_index[2] = i + 11;
      game.ghost[i].state        = GHOST_STATE_ANGRY;
      game.ghost[i].move_seq_id  = 0;

      game.ghost[i].remove_extra_pixel_move    = false;

      console_sprite_set_values 
         (game.ghost[i].sprite_index[0], 'p', ghost_get_color (i), 7 + 10 * 8, (13 + i) * 8, 0, 0);
      console_sprite_set_values 
         (game.ghost[i].sprite_index[1], 118, COLOR_WHITE, 7 + 10 * 8, (13 + i) * 8, 0, 0);
      console_sprite_set_values 
         (game.ghost[i].sprite_index[2], 120, COLOR_DKBLUE, 7 + 10 * 8, (13 + i) * 8, 0, 0);
   }

}

// draws a score on the display
void score_draw_one (int id, int score) {
   char *s = dylib.int2str (score);
   int sl  = dylib.strlen (s);
   int pos;
   switch (id) {
      case 0:
         pos = 9 - sl;
         break;
      case 1:
         pos = 25 - sl;
         break;
      default:
         pos = 17 - sl;
         break;
   }
   console_write_raw (1, pos, s, sl);
}

// draws all scores on the display
void score_draw_all () {
   score_draw_one (0, game.player[0].score);
   score_draw_one (1, game.player[1].score);
   score_draw_one (2, game.high_score);
}

// displays the treats eaten
void treats_display_eaten () {

   const int treats_to_display = min (game.player[game.player_id].treat.treat_index, 8); // cap at 8, since the next 8 are repeats

   if (treats_to_display) { // write to the display only if at least one treat has been eaten
      console_write_raw (23, 27 - treats_to_display, (char *)treat_char_id, treats_to_display);
   }
}

// displays the number of pacman lives remaining (less the current one in play)
void lives_display_count () {
      
   // array of pacman defs to display
   const unsigned char s[] = {
      pacman_move_char_id[1][3],
      pacman_move_char_id[1][3], 
      pacman_move_char_id[1][3], 
      pacman_move_char_id[1][3],  
      pacman_move_char_id[1][3], 
      pacman_move_char_id[1][3]
   };
              
   // number of lives to display - this is one less than remaining since one is on the play field
   const int lives_to_display = game.player[game.player_id].lives_remaining - 1;
   
   // lives to display may be zero
   if (lives_to_display) {
      // write to the last row, starting under the playfield
      console_write_raw (23, 4, (char *)s, lives_to_display);
   }
}  

// adds score to the player, update high score, displays scores as necessary and adds extra play if needed
void player_add_score (int v) {
   game.player[game.player_id].score += v;
   score_draw_one (game.player_id, game.player[game.player_id].score);
   if (game.player[game.player_id].score > game.high_score) {
      game.high_score = game.player[game.player_id].score;
      score_draw_one (-1, game.high_score);
   }

   if (game.player[game.player_id].score >= game.player[game.player_id].extra_life_score) {
      game.player[game.player_id].lives_remaining++;
      lives_display_count ();
      game.player[game.player_id].extra_life_score = 32767;
   }
}

// clears a single position in the playfield in ram
void playfield_clear_position (int y, int x) {
   game.player[game.player_id].play_field[y][x] = ' ';
// VDP_SET_ADDRESS_WRITE (y * 32 + x);
// VDPWD = game.player[game.player_id].play_field[y][x];
   console_write_raw (y, x, &game.player[game.player_id].play_field[y][x], 1);
}

// moves the pacman
void pacman_move () {
   static bool remove_eaten_powerpellet = true;
   bool is_aligned;
   bool did_teleport = false;
   int y, x, py, px;
   int scared_level;
   int pacman_vel_y, pacman_vel_x;
   int move_multiplier;

   int j;

   // delay based on speed
   sequencer_set_throttle (game.speed << 5);

   int k = console_get_key ();
   console_sprite_test_char_alignment (game.pacman.sprite_index, &is_aligned, &y, &x);
   if (is_aligned) {
      switch (game.player[game.player_id].play_field[y][x]) {
         case '$': // handle the teleport from the end of the left cooridor to the right
            // get the current position
            console_sprite_get_position (game.pacman.sprite_index, &py, &px);
            // shift right by 22 * 8 pixels
            console_sprite_set_position (game.pacman.sprite_index, py, px + 22 * 8);
            // adjust the x value by 22 characters to the right
            x += 22;
            did_teleport = true;
            break;
         case '#': // handle the teleport from the end of the right cooridor to the left
            // get the current position
            console_sprite_get_position (game.pacman.sprite_index, &py, &px);
            // shift left by 22 * 8 pixels
            console_sprite_set_position (game.pacman.sprite_index, py, px - 22 * 8);
            // adjust the x value by 22 characters to the left
            x -= 22;
            did_teleport = true;
            break;
         case '^': // handle eating power pellet
            if (remove_eaten_powerpellet) {
               playfield_clear_position (y, x);
            }
            for (j = 0; j < NUM_GHOSTS; j++) {
               game.ghost[j].state         = GHOST_STATE_SCARED;
               scared_level = game.player[game.player_id].level;
               if (scared_level > 15) scared_level = 15;
               game.ghost[j].state_timeout = ghost_scared_time_for_level[scared_level];
               game.ghost[j].move_seq_id   = GHOST_CHASE_SEQ_FIRST;
            }
            player_add_score (10);                     // add to player's score
            game.bonus.value_index = BONUS_200;  // set the bonus for the first ghost to be eaten
            game.player[game.player_id].pellets_remaining--;
            game.pacman.state_timeout = ghost_scared_time_for_level[scared_level];
//          game.pacman.state_timeout = 50;
            break;
         case '!': // handle eating a treat
            player_add_score (treat_bonus_val[game.player[game.player_id].treat.treat_index]);
            console_sprite_get_position (game.pacman.sprite_index, &py, &px);
            treat_bonus_show (game.player[game.player_id].treat.treat_index, py, px);
            playfield_clear_position (y, x);
            game.player[game.player_id].treat.treat_index++;
            if (game.player[game.player_id].treat.treat_index > 15) {
               game.player[game.player_id].treat.treat_index = 15;
            }
            treats_display_eaten ();
            sound_play_gobble ();
            break;
         case '_': // handle eating a regular pellet
            playfield_clear_position (y, x);
            player_add_score (1);
            game.player[game.player_id].pellets_remaining--;
            sound_play_gobble ();
            break;
         default:
            break;
      }

      if (!did_teleport) {

         // check for joystick and map to keys
         int joy_y, joy_x;
         console_joystick_read (game.player_id, &joy_y, &joy_x);
         if (joy_y) {
            if (joy_y == JOYSTICK_UP) {
               k = KEY_UP;
            } else {
               k = KEY_DOWN;
            }
         } else if (joy_x) {
            if (joy_x == JOYSTICK_LEFT) {
               k = KEY_LEFT;
            } else {
               k = KEY_RIGHT;
            }
         }
         switch (k) {
            case 'p':
               break;
            case KEY_UP:
               if (!(game.player[game.player_id].play_field[y - 1][x] >= PLAYFIELD_BORDER_CHAR_FIRST &&
                     game.player[game.player_id].play_field[y - 1][x] <= PLAYFIELD_BORDER_CHAR_LAST) &&
                   game.player[game.player_id].play_field[y - 1][x] != PLAYFIELD_BORDER_PEN_DOOR_CHAR) {
                  game.pacman.dir = 0;
               }
               break;
            case KEY_RIGHT:
               if (!(game.player[game.player_id].play_field[y][x + 1] >= PLAYFIELD_BORDER_CHAR_FIRST &&
                     game.player[game.player_id].play_field[y][x + 1] <= PLAYFIELD_BORDER_CHAR_LAST) &&
                   game.player[game.player_id].play_field[y][x + 1] != PLAYFIELD_BORDER_PEN_DOOR_CHAR) {
                  game.pacman.dir = 1;
               }
               break;
            case KEY_DOWN:
               if (!(game.player[game.player_id].play_field[y + 1][x] >= PLAYFIELD_BORDER_CHAR_FIRST &&
                     game.player[game.player_id].play_field[y + 1][x] <= PLAYFIELD_BORDER_CHAR_LAST) &&
                   game.player[game.player_id].play_field[y + 1][x] != PLAYFIELD_BORDER_PEN_DOOR_CHAR) {
                  game.pacman.dir = 2;
               }
               break;
            case KEY_LEFT:
               if (!(game.player[game.player_id].play_field[y][x - 1] >= PLAYFIELD_BORDER_CHAR_FIRST &&
                     game.player[game.player_id].play_field[y][x - 1] <= PLAYFIELD_BORDER_CHAR_LAST) &&
                   game.player[game.player_id].play_field[y][x - 1] != PLAYFIELD_BORDER_PEN_DOOR_CHAR) {
                  game.pacman.dir = 3;
               }
               break;
            case '9':
               remove_eaten_powerpellet = false;
               break;
            case '0':
               remove_eaten_powerpellet = true;
               break;
            default:
               break;
         }
      }

      switch (game.pacman.dir) {
         case 0:
            if ((game.player[game.player_id].play_field[y - 1][x] >= PLAYFIELD_BORDER_CHAR_FIRST &&
                 game.player[game.player_id].play_field[y - 1][x] <= PLAYFIELD_BORDER_CHAR_LAST) ||
                game.player[game.player_id].play_field[y - 1][x] == PLAYFIELD_BORDER_PEN_DOOR_CHAR) {
               pacman_vel_y = 0;
               pacman_vel_x = 0;
            } else {
               pacman_vel_y = -1;
               pacman_vel_x = 0;
            }
            break;
         case 1:
            if ((game.player[game.player_id].play_field[y][x + 1] >= PLAYFIELD_BORDER_CHAR_FIRST &&
                 game.player[game.player_id].play_field[y][x + 1] <= PLAYFIELD_BORDER_CHAR_LAST)  ||
                game.player[game.player_id].play_field[y][x + 1] == PLAYFIELD_BORDER_PEN_DOOR_CHAR) {
               pacman_vel_y = 0;
               pacman_vel_x = 0;
            } else {
               pacman_vel_y = 0;
               pacman_vel_x = 1;
            }
            break;
         case 2:
            if ((game.player[game.player_id].play_field[y + 1][x] >= PLAYFIELD_BORDER_CHAR_FIRST &&
                 game.player[game.player_id].play_field[y + 1][x] <= PLAYFIELD_BORDER_CHAR_LAST) ||
                game.player[game.player_id].play_field[y + 1][x] == PLAYFIELD_BORDER_PEN_DOOR_CHAR) {
               pacman_vel_y = 0;
               pacman_vel_x = 0;
            } else {
               pacman_vel_y = 1;
               pacman_vel_x = 0;
            }
            break;
         case 3:
            if ((game.player[game.player_id].play_field[y][x - 1] >= PLAYFIELD_BORDER_CHAR_FIRST &&
                 game.player[game.player_id].play_field[y][x - 1] <= PLAYFIELD_BORDER_CHAR_LAST) ||
                game.player[game.player_id].play_field[y][x - 1] == PLAYFIELD_BORDER_PEN_DOOR_CHAR) {
               pacman_vel_y = 0;
               pacman_vel_x = 0;
            } else {
               pacman_vel_y = 0;
               pacman_vel_x = -1;
            }
            break;
         default:
            pacman_vel_y = 0;
            pacman_vel_x = 0;
            break;
      }
      game.pacman.image_index = 0;

      game.pacman.state_timeout--;                                               // decrement state timeout
      if (game.pacman.state_timeout <= 0) {                                      // limit range to 0+
         game.pacman.state_timeout = 0;
         move_multiplier           = 1;                                          // set move multipler to standard rate
      } else {                                                                   // otherwise the pacman is in power-pellet mode
         if (pacman_vel_y != 0 || pacman_vel_x != 0) {                           // if the pacman is moving,
            move_multiplier                     = PACMAN_EXTRA_PIXEL_MULTIPLIER; // set the move multipler to boosted and
            game.pacman.remove_extra_pixel_move = true;                          // set to slow down on next move
         } else {
            move_multiplier                     = 1;                             // default the move multiplier - not moving anyway
         }
      }
   
      console_sprite_set_motion (game.pacman.sprite_index, pacman_vel_y * move_multiplier, pacman_vel_x * move_multiplier);
   } else {
      if (game.pacman.remove_extra_pixel_move) {
         console_sprite_get_values (game.pacman.sprite_index, &py, &px, &pacman_vel_y, &pacman_vel_x);
         pacman_vel_y = pacman_vel_y / PACMAN_EXTRA_PIXEL_MULTIPLIER;
         pacman_vel_x = pacman_vel_x / PACMAN_EXTRA_PIXEL_MULTIPLIER;
         console_sprite_set_motion (game.pacman.sprite_index, pacman_vel_y, pacman_vel_x);
         game.pacman.remove_extra_pixel_move = false;
      }
   }

   console_sprite_set_char (game.pacman.sprite_index, pacman_move_char_id[game.pacman.dir][game.pacman.image_index]);
   game.pacman.image_index++;
}

// tests the pacman and ghosts for coincidences
void pacman_check_ghost_coincidence () {
   bool c;
   int y, x, vy, vx;
   for (int i = 0; i < NUM_GHOSTS; i++) {
      c = console_sprite_coincidence (game.pacman.sprite_index, game.ghost[i].sprite_index[0], 5);
      if (c) {
         switch (game.ghost[i].state) {
            case GHOST_STATE_ANGRY:
               soundqueue_honk ();
               game.player_is_alive = false;
               break;
            case GHOST_STATE_SCARED:
               sound_play_ghost_eaten ();
               game.ghost[i].state       = GHOST_STATE_EATEN;
               game.ghost[i].move_seq_id = GHOST_RETURN_TO_HOME_MOVE_SEQ_FIRST;
               player_add_score (ghost_bonus_val[game.bonus.value_index]);                  // add the bonus to the score
               console_sprite_set_char (game.ghost[i].sprite_index[0], 0);                  // make the ghost tempoarily disappear
               console_sprite_get_values (game.ghost[i].sprite_index[0], &y, &x, &vy, &vx); // get the ghost position
               ghost_bonus_show (game.bonus.value_index, y, x);                             // add the flying bonus
               game.bonus.value_index++;                                                    // set the next bonus level
               console_sprite_disable_motion ();                                            // disable motion until the sound is complete
               delay_until_sound_end ();                                                    // wait until sound completes
               console_sprite_enable_motion ();                                             // enable motion
               break;
            default:
               break;
         }
      }
   }
}

// returns true if the path is open in the play field -- only used for ghosts as it doesn't test for the door
bool playfield_is_path_open (int x, int y) {
  return !(game.player[game.player_id].play_field[y][x] >= PLAYFIELD_BORDER_CHAR_FIRST &&
           game.player[game.player_id].play_field[y][x] <= PLAYFIELD_BORDER_CHAR_LAST);
}

// determines a ghost's next target position (where to go)
void ghost_get_target_position (int id, int *y, int *x) {

   int yt, xt, vyt, vxt;

   // determine where this ghost should be drawn to when angry, or repulsed by when scared
   switch (game.ghost[id].state) {
      case GHOST_STATE_ANGRY:
      case GHOST_STATE_SCARED:
         switch (game.ghost[id].move_seq_id) {
            case GHOST_HOLD_MOVE_SEQ_FIRST ... GHOST_HOLD_MOVE_SEQ_LAST:
               *y = ghost_house_hold_center_y[id];
               *x = ghost_house_hold_center_x[id];
               break;
            case GHOST_HOUSE_EXIT_MOVE_SEQ_FIRST ... GHOST_HOUSE_EXIT_MOVE_SEQ_LAST:
               *y = ghost_house_exit_center_y[id];
               *x = ghost_house_exit_center_x[id];
               break;
            case GHOST_OWN_AREA_SEQ_FIRST ... GHOST_OWN_AREA_SEQ_LAST:
               *y = ghost_haunt_area_center_y[id];
               *x = ghost_haunt_area_center_x[id];
               break;
            case GHOST_CHASE_SEQ_FIRST ... GHOST_CHASE_SEQ_LAST:
               console_sprite_get_values (game.pacman.sprite_index, &yt, &xt, &vyt, &vxt);
               *y = yt;
               *x = xt;
               break;
            case GHOST_RESET_MOVE_SEQ:
               *y = ghost_house_exit_center_y[id];
               *x = ghost_house_exit_center_x[id];
               game.ghost[id].move_seq_id = GHOST_OWN_AREA_SEQ_FIRST;
               break;
            default:
               break;
         }
         break;
      case GHOST_STATE_EATEN:
         switch (game.ghost[id].move_seq_id) {
            case GHOST_RETURN_TO_HOME_MOVE_SEQ_FIRST ... GHOST_RETURN_TO_HOME_MOVE_SEQ_LAST:
               *y = ghost_house_hold_center_y[id];
               *x = ghost_house_hold_center_x[id];
               break;
            default:
               *y = ghost_house_hold_center_y[id];
               *x = ghost_house_hold_center_x[id];
               game.ghost[id].move_seq_id = GHOST_OWN_AREA_SEQ_FIRST;
               game.ghost[id].state       = GHOST_STATE_ANGRY;
               break;
         }
      default:
         break;
   }
   game.ghost[id].move_seq_id++;
}

// moves the ghosts
void ghosts_move () {

   static int pattern_id = 0; // the pattern id, in range 0-1

   bool is_aligned;
   bool move_selected;

   int one;

   int seek_pos_y, seek_pos_x;
   int ghost_pos_y, ghost_pos_x, ghost_vel_y, ghost_vel_x;
   int ghost_grid_y, ghost_grid_x;
   int move_multiplier;

   // advance the pattern id
   pattern_id++;
   if (pattern_id > 1) {
      pattern_id = 0;
   }

   // loop through all the ghosts
   for (int k = 0; k < NUM_GHOSTS; k++) {

      // set the ghost char def based on ghost state and pattern id
      console_sprite_set_char (game.ghost[k].sprite_index[0], ghost_char_id[game.ghost[k].state][pattern_id]);

      // set the ghost color based on its state
      console_sprite_set_color (game.ghost[k].sprite_index[0], ghost_get_color (k));

      // determine if sprite is aligned on the character grid
      console_sprite_test_char_alignment (game.ghost[k].sprite_index[0], &is_aligned, &ghost_grid_y, &ghost_grid_x); 
      if (is_aligned) {

         // gets the ghost's current position and velocity
         console_sprite_get_values (game.ghost[k].sprite_index[0], &ghost_pos_y, &ghost_pos_x, &ghost_vel_y, &ghost_vel_x);

         // process the character at this aligned position
         switch (game.player[game.player_id].play_field[ghost_grid_y][ghost_grid_x]) {
            case '$': // handle the teleport from the end of the left cooridor to the right
               // shift right by 22 * 8 pixels
               ghost_pos_x += 22 * 8;
               console_sprite_set_position (game.ghost[k].sprite_index[0], ghost_pos_y, ghost_pos_x);
               console_sprite_set_position (game.ghost[k].sprite_index[1], ghost_pos_y, ghost_pos_x);
               console_sprite_set_position (game.ghost[k].sprite_index[2], ghost_pos_y, ghost_pos_x);
               // adjust the x value by 22 characters to the right
               ghost_grid_x += 22;
               break;
            case '#': // handle the teleport from the end of the right cooridor to the left
               // shift left by 22 * 8 pixels
               ghost_pos_x -= 22 * 8;
               console_sprite_set_position (game.ghost[k].sprite_index[0], ghost_pos_y, ghost_pos_x);
               console_sprite_set_position (game.ghost[k].sprite_index[1], ghost_pos_y, ghost_pos_x);
               console_sprite_set_position (game.ghost[k].sprite_index[2], ghost_pos_y, ghost_pos_x);
               // adjust the x value by 22 characters to the left
               ghost_grid_x -= 22;
               break;
            default:
               break;
         }

         // determine where this ghost should be drawn to when angry, or repulsed by when scared
         ghost_get_target_position (k, &seek_pos_y, &seek_pos_x);

         if (game.ghost[k].state == GHOST_STATE_SCARED) {
            // time out the ghost being scared
            game.ghost[k].state_timeout--;
            if (!game.ghost[k].state_timeout) {
               game.ghost[k].state = GHOST_STATE_ANGRY;
            }
         }

         if (game.ghost[k].state == GHOST_STATE_SCARED) {
            one = -1;
         } else {
            one = 1;
         }
            
         move_selected = false;
   
         // SELECT LEFT
         if (!move_selected &&
             ghost_vel_x == 0 &&
             ghost_pos_x > seek_pos_x && 
             playfield_is_path_open (ghost_grid_x - one, ghost_grid_y)) {

            ghost_vel_x   = -one;
            ghost_vel_y   = 0;
            move_selected = true;

            // set the ghost eyes direction
            console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
         }
   
         // SELECT RIGHT
         if (!move_selected &&
             ghost_vel_x == 0 &&
             ghost_pos_x < seek_pos_x && 
             playfield_is_path_open (ghost_grid_x + one, ghost_grid_y)) {
   
            ghost_vel_x   = one;
            ghost_vel_y   = 0;
            move_selected = true;

            // set the ghost eyes direction
            console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
         }
   
         // SELECT UP
         if (!move_selected &&
             ghost_vel_y == 0 &&
             ghost_pos_y > seek_pos_y && 
             playfield_is_path_open (ghost_grid_x, ghost_grid_y - one)) {
   
            ghost_vel_x   = 0;
            ghost_vel_y   = -one;
            move_selected = true;

            // set the ghost eyes direction
            console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
         }

         // SELECT DOWN
         if (!move_selected &&
             ghost_vel_y == 0 &&
             ghost_pos_y < seek_pos_y && 
             playfield_is_path_open (ghost_grid_x, ghost_grid_y + one)) {

            ghost_vel_x   = 0;
            ghost_vel_y   = one;
            move_selected = true;

            // set the ghost eyes direction
            console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
         }

         // CONTINUE ON SELECTED PATH
         if (!move_selected &&
             (ghost_vel_y + ghost_vel_x != 0) &&
             playfield_is_path_open (ghost_grid_x + ghost_vel_x, ghost_grid_y + ghost_vel_y)) {

            move_selected = true;
         }
   
         if (ghost_vel_y != 0) {
            // TRY LEFT
            if (!move_selected &&
                playfield_is_path_open (ghost_grid_x - one, ghost_grid_y)) {
  
                ghost_vel_y   = 0;
                ghost_vel_x   = -one;
                move_selected = true;

                // set the ghost eyes direction
                console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
            }

            // TRY RIGHT
            if (!move_selected &&
                playfield_is_path_open (ghost_grid_x + one, ghost_grid_y)) {
  
                ghost_vel_y   = 0;
                ghost_vel_x   = one;
                move_selected = true;

                // set the ghost eyes direction
                console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
            }

            // TRY UP     
            if (!move_selected &&
                playfield_is_path_open (ghost_grid_x, ghost_grid_y - one)) {
  
                ghost_vel_y   = -one;
                ghost_vel_x   = 0;
                move_selected = true;

                // set the ghost eyes direction
                console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
            }

            // TRY DOWN
            if (!move_selected &&
                playfield_is_path_open (ghost_grid_x, ghost_grid_y + one)) {
  
                ghost_vel_y   = one;
                ghost_vel_x   = 0;
                move_selected = true;

                // set the ghost eyes direction
                console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
            }
         } else {

            // TRY UP     
            if (!move_selected &&
                playfield_is_path_open (ghost_grid_x, ghost_grid_y - one)) {
  
                ghost_vel_y   = -one;
                ghost_vel_x   = 0;
                move_selected = true;

                // set the ghost eyes direction
                console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
            }

            // TRY DOWN
            if (!move_selected &&
                playfield_is_path_open (ghost_grid_x, ghost_grid_y + one)) {
  
                ghost_vel_y   = one;
                ghost_vel_x   = 0;
                move_selected = true;

                // set the ghost eyes direction
                console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
            }

            // TRY LEFT
            if (!move_selected &&
                playfield_is_path_open (ghost_grid_x - one, ghost_grid_y)) {
  
                ghost_vel_y   = 0;
                ghost_vel_x   = -one;
                move_selected = true;

                // set the ghost eyes direction
                console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
            }

            // TRY RIGHT
            if (!move_selected &&
                playfield_is_path_open (ghost_grid_x + one, ghost_grid_y)) {
  
                ghost_vel_y   = 0;
                ghost_vel_x   = one;
                move_selected = true;

                // set the ghost eyes direction
                console_sprite_set_char (game.ghost[k].sprite_index[1], ghost_eye_char_id[ghost_vel_y + 1][ghost_vel_x + 1]);
            }

         }

         if (game.ghost[k].state == GHOST_STATE_ANGRY || game.ghost[k].state == GHOST_STATE_EATEN) {
            move_multiplier = GHOST_EXTRA_PIXEL_MULTIPLER;
            game.ghost[k].remove_extra_pixel_move = true;
         } else {
            move_multiplier = 1;
         }

         ghost_vel_y *= move_multiplier;
         ghost_vel_x *= move_multiplier;

         console_sprite_set_motion (game.ghost[k].sprite_index[0], ghost_vel_y, ghost_vel_x);
         console_sprite_set_motion (game.ghost[k].sprite_index[1], ghost_vel_y, ghost_vel_x);
         console_sprite_set_motion (game.ghost[k].sprite_index[2], ghost_vel_y, ghost_vel_x);

      } else {
         if (game.ghost[k].remove_extra_pixel_move) {
            console_sprite_get_values (game.ghost[k].sprite_index[0], &ghost_pos_y, &ghost_pos_x, &ghost_vel_y, &ghost_vel_x);
            ghost_vel_y = ghost_vel_y / GHOST_EXTRA_PIXEL_MULTIPLER;
            ghost_vel_x = ghost_vel_x / GHOST_EXTRA_PIXEL_MULTIPLER;
            console_sprite_set_motion (game.ghost[k].sprite_index[0], ghost_vel_y, ghost_vel_x);
            console_sprite_set_motion (game.ghost[k].sprite_index[1], ghost_vel_y, ghost_vel_x);
            console_sprite_set_motion (game.ghost[k].sprite_index[2], ghost_vel_y, ghost_vel_x);
            game.ghost[k].remove_extra_pixel_move = false;
         }
      }
   }
}

// initializes the treat for the beginning of the game
void treat_init (int player_id) {
   game.player[player_id].treat.treat_index = 0;    // initialize the treat index
   game.player[player_id].play_field[TREAT_POS_Y][TREAT_POS_X] = 32;
}

// initializes the treat sequence for the level
void treat_init_wave (int player_id) {
   game.player[player_id].treat.state_seq_id                   = 0;
   game.player[player_id].play_field[TREAT_POS_Y][TREAT_POS_X] = 32;
}

// clears the treat
void treat_clear () {
   playfield_clear_position (TREAT_POS_Y, TREAT_POS_X);
}

// moves the treat (really just advances the treat seqeunce)
void treat_move (int player_id) {
   unsigned char s[2];
   switch (game.player[player_id].treat.state_seq_id) {
      case 1024: // turn on treat
         // note that the character added to the play_field array is 33 but the display is written a different value (confusing)
         game.player[player_id].play_field[TREAT_POS_Y][TREAT_POS_X] = 33;
         s[0] = treat_char_id[game.player[player_id].treat.treat_index];
         console_write_raw (TREAT_POS_Y, TREAT_POS_X, (char *) s, 1);
         break;
      case 2048: // turn off treat, whether eaten or not, and advance to the next treat
         playfield_clear_position (TREAT_POS_Y, TREAT_POS_X);
         break;
      case 32767: // prevent overflow on state_seq_id int that would cause the next treat to appear :-)
         game.player[player_id].treat.state_seq_id = 2048; // will skip the previous case
         break;
      default:
         break;
   }
   game.player[player_id].treat.state_seq_id++;
}

// calls the player to the game -- primarily when the player changes
void display_player_call (int player_id) {
   char pl_call[16];
   dylib.strcpy (pl_call, "PLAYER ");
   dylib.strcat (pl_call, dylib.int2str (game.player_id + 1));
   console_write_raw (9, 11, pl_call, dylib.strlen (pl_call));
}

// displays game over
void display_player_game_over () {
   const char game_over[] = "GAME OVER"; 
   console_write_raw (13, 11, game_over, dylib.strlen (game_over));
}

// calls the player to be ready
void display_ready_call () {
   const unsigned char expfont[8] = {0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00};
   const char *ready = "READY!";
   console_font_set (33, expfont);
   console_write_raw (13, 12, ready, dylib.strlen (ready));
}

// delays game play while continuing to advance the sequencer
void delay_do (int c) {
   sequencer_set_throttle (256);
   for (int i = 0; i < c; i++) {
      console_get_key ();
   }
   sequencer_set_throttle (0);
}

// flashes the play field upon successful clearing of pellets
void playfield_flash_clear_success () {

   int i, j;

   delay_do (16);

   for (i = 0; i < 3; i++) {
      for (j = 12; j < 16; j++) {
         console_standard_set_char_group_color (j, COLOR_WHITE, COLOR_TRANS);
      }

      delay_do (16);

      for (j = 12; j < 16; j++) {
         console_standard_set_char_group_color (j, COLOR_DKBLUE, COLOR_TRANS);
      }

      delay_do (16);
   }
}

// perform the pacman death sequence
void pacman_die () {

   delay_do (16);

   ghosts_hide ();

   for (int i = 0; i < PACMAN_DEATH_NUM_SEQ; i++) {
      console_sprite_set_char (game.pacman.sprite_index, pacman_death_char_id[i]);
      delay_do (16);
   }

   entities_hide ();
   delay_do (16);
}

// initializes game values
void game_init_values (int num_players) {
   game.num_players   = num_players;           // set the number of players
   game.player_id     = 0;                     // set the player id to zero
   for (int i = 0; i < 2; i++) {               // loop through both players
      game.player[i].score             = 0;    // initialize the score
      game.player[i].extra_life_score  = 1000; // set the extra life score required
      game.player[i].pellets_remaining = 0;    // initialize the number of pellets remaining on the screen
      game.player[i].is_start_of_game  = true; // set as start of name
      game.player[i].level             = -1;   // set the level
      treat_init (i);                          // initialize the treats
   }
   game.player[0].lives_remaining    = 3;      // initialize the number of lives remaining for the first player
   if (num_players == 2) {                           // initialize the number of lives remaining for the second player
      game.player[1].lives_remaining = 3;      // two player game so set to 3
   } else {
      game.player[1].lives_remaining = 0;      // one player game so set to 0
   }
}

// plays the game
void game_play (int num_players) {
   int last_player_id = UNDEFINED;

   game_init_values (num_players);

   // continue playing until there's no more lives remaining
   while (game.player[0].lives_remaining + game.player[1].lives_remaining != 0) {
      // play this player if it has lives remaining
      if (game.player[game.player_id].lives_remaining) {
         entities_hide ();                                                  // hide all the characters
         game.player_is_alive = true;                                       // set this player as alive
         bool is_first_cycle = true;                                        // set this as the first display cycle
         while (game.player_is_alive) {                                     // loop while the player is alive
            if (game.player[game.player_id].pellets_remaining == 0) {       // this is the stage/wave transition
               game.player[game.player_id].level++;
               if (!game.player[game.player_id].is_start_of_game) {         // if not the start of the game
                  entities_stop ();                                         // stop the pacman and ghosts
                  playfield_flash_clear_success ();                         // flash the screen for success
               }
               game.player[game.player_id].pellets_remaining = 178;         // set the count back to the top
               playfield_init (game.player_id);                             // initialize the players play field
               treat_init_wave (game.player_id);                            // initialize the treats for this wave
               is_first_cycle = true;                                       // set this as the first display cycle
            }
            if (is_first_cycle) {                                           // do everything for the first cycle
               treat_clear ();                                              // clear the treat
               playfield_draw (true);                                       // draw the current play field
               lives_display_count ();                                      // draw the number of pacmen
               treats_display_eaten ();
               score_draw_all ();                                           // draw the scores
	       entities_init ();                                            // initialize all entities
               display_ready_call ();                                       // display the ready call
               if (game.player[game.player_id].is_start_of_game) {          // handle start of game
                  display_player_call (game.player_id);                     // call the player
                  if (game.player_id == 1) {                                // wait before playing music if this is the 2nd player
                     delay_do (256);
                  }
                  sound_play_startup_music ();                              // play the startup music
                  delay_until_sound_end ();                                 // wait until the music is over
                  game.player[game.player_id].is_start_of_game = false;     // mark the end of the start of game
               } else {
                  if (last_player_id != game.player_id) {                   // if this player is different then last then call
                     display_player_call (game.player_id);                  // the other user back
                  }
                  delay_do (256);                                           // wait a bit
               }
               playfield_draw (false);                                      // redraw the field
               score_draw_all ();                                           // draw the scores
               lives_display_count ();                                      // draw the number of pacmen
               treats_display_eaten ();
               is_first_cycle = false;                                      // conclude the first cycle
            }
            console_reset_blanking ();                                      // prevent screen from blanking due to time out
            pacman_move ();                                                 // move the pacman
            pacman_check_ghost_coincidence ();                              // test if the pacman has coincided with ghosts
            ghosts_move ();                                                 // move the ghosts
            treat_move (game.player_id);                                    // update the treat
         }
         entities_stop ();                                                  // hide all the characters
         delay_until_sound_end (256);                                       // wait until all sound is done playing
         pacman_die ();
         game.player[game.player_id].lives_remaining--;                     // decrement the number of lives remaining
         if (!game.player[game.player_id].lives_remaining) {                // check for game over
            display_player_game_over ();                                    // display game over
            delay_do (256);                                                 // wait a bit
         }
      }
      last_player_id = game.player_id;                                      // save the last player id
      game.player_id++;                                                     // increment the player id
      if (game.player_id >= num_players) {                                  // make sure the player number doesn't exceed the 
                                                                            // number of players
         game.player_id = 0;                                                // reset back to the first player
      }
   }
   entities_hide ();                                                        // hide all the characters
}

// draws the game select screen
void gameselect_draw () {
   dylib.fputs ("\f", stdout);
   for (int i = 0; i < 2; i++) {
      dylib.fputs (template_play_field[i], stdout);
   }
   dylib.fputs ("\n\n\n\n\n\n", stdout);
   dylib.fputs ("       PUSH START BUTTON       \n\n\n", stdout);
   dylib.fputs ("        1 OR 2 PLAYERS         \n\n\n", stdout);
   dylib.fputs ("  BONUS PAC-MAN FOR 10000 PTS  \n\n\n", stdout);
   dylib.fputs ("  @2024 VAN ELECTRONICS, INC.  \n\n\n", stdout);
   dylib.fputs ("        PUSH X TO EXIT         \n\n\n", stdout);
   dylib.fputs ("  CREDIT  2", stdout);
   score_draw_all ();
}

// initializes the program
void prog_init (const char *argv0, int *recover_display_mode, int *recover_display_rows, int speed) {
   console_display_get_mode (recover_display_mode, recover_display_rows);
   console_display_set_mode (DISPLAY_MODE_STANDARD, DISPLAY_ROWS_24);
   console_cls ();
   dylib.fputs ("\n\n\n\n\n\n\n\n\n\n\n\t\t   PACMAN", stdout);
   colors_init ();
   chardefs_init (argv0);
   memset (&game, 0, sizeof (game_t)); // should initialize all values to make the entire program happy
   game.speed = speed;
   dylib.fputs ("\f", stdout);
   
   sequencer_add (powerpellets_blink_service);    // add power pellet blinking
   sequencer_add (playerid_blink_service);        // add player id blinking
   bonus_init ();                                 // initialize the bonus info
   sequencer_add (bonus_display_service);         // add bonus display
} 

// runs the program
void prog_run () {
   bool continue_running = true;        // whether to continue running
   bool draw_game_select = true;
   int k;
   while (continue_running) {           // loop until exit
      if (draw_game_select) {
         gameselect_draw ();               // display selection screen
         draw_game_select = false;
      }
      k = console_getc ();              // wait for and get a key press
      switch (k) {                      // act upon the key press
         case '1':                      // play the game with 1 player
            game_play (1);  
            draw_game_select = true;
            break;
         case '2':                      // play the game with 2 players
            game_play (2);
            draw_game_select = true;
            break;
         case 'X':                      // stop the game
         case 'x':
            continue_running = false;
         default:
            break;
      }
   }
}

// mode and rows to return to at program termination
int recover_display_mode;
int recover_display_rows;

// the program atexit procedure
void prog_atexit () {
   prog_reset_display (recover_display_mode, recover_display_rows);     // reset the display mode, colors and fonts
}

// the main program
int main (int argc, char *argv[]) {

   bool error = false;

   int speed = 0;

   // capture the options
   int opt;
   while ((opt = getopt (argc, argv, "s:")) != -1) {                       // loop through all options
      switch (opt) {                                                       // switch on an option
         case 's':                                                         // capture specified field delimiter
            speed = atoi (optarg);                                         // set the value
            if (speed < 0 ||                                               // test the value in range
                speed > 20) {

               dylib.fputs ("pacman: illegal speed\n", stderr);            // print an error message
               error = true;                                               // set the error flag
            }
            break;
         case '?':
         default:
            error = true;
            break;
      }
   }

   if (!error) {
      prog_init                                                            // initialize the program
         (argv[0], 
          &recover_display_mode, 
          &recover_display_rows, 
          speed);  
      dylib.atexit (prog_atexit);                                          // set the atexit method, which will nicely clean up
      prog_run ();                                                         // run the program
   } else {
      dylib.fputs ("usage: pacman [-s speed]\n", stderr);                  // print an error message
   }

   return 0;                                                               // return success
}
