/* passwd.c
   This program updates the password in the password file.
   Typical passwd command prompts and responses:
      Changing password for marko.
      Current password: 
      New password: 
      Retype new password: 
      The password has not been changed.
      New password: 
      Password change has been aborted.
      passwd: Authentication token manipulation error
      passwd: password unchanged
   change log:
   11/11/2023 initial version
   11/12/2023 updated to use get_passwd_file_path
              modified to allow root to change any other user's password without requiring entry of the current password
   11/13/2023 updated to use passwd_get_value instead of passwd_get_cipher
   11/21/2023 updated to use getpass
   02/27/2024 added use of dylib
   01/20/2025 updated fputs to dylib
*/

#include <stdio.h>
#include <cipher.h>
#include <string.h>
#include <unistd.h>
#include <unistd_private.h>
#include <stdlib.h>
#include <dylib.h>

// the passwd main method
int main (int argc, char *argv[]) {

   char username[10];  
   char password[10];
   char password_repeat[10];
   char cipher[32];
   char cipher_check[32];
   int r;
   bool skip_old_password_check;

   if (argc == 2) {                                                               // test if a username was specified as an argument
      dylib.strcpy (username, argv[1]);                                           // capture the username argument
   } else {
      dylib.strcpy (username, getlogin ());                                       // capture the current user by default
   }

   if (passwd_get_value (username, PASSWD_CIPHER, cipher)) {                      // get the specified user's password cipher
      dylib.fputs ("Changing password for ", stdout);                             // prompt the user
      dylib.fputs (username, stdout);
      dylib.fputs (".\n", stdout);

      skip_old_password_check =                                                   // determine if the old password check should be
         !dylib.strcmp (getlogin (), "root") && dylib.strcmp (username, "root");  // skipped

      if (!skip_old_password_check) {
         dylib.strcpy (password, getpass ("Old Password: "));                     // prompt for and gtet old password
         cipher_vic_encrypt (password, cipher_check, sizeof (cipher_check));      // generate the cipher for the old password
      } else {
         dylib.strcpy (cipher_check, "0123456789ABCDEF");
      }

      if (!dylib.strcmp (cipher, cipher_check) || skip_old_password_check) {      // confirm they are identical
         dylib.strcpy (password, getpass ("New Password: "));                     // prompt for and get new password
         dylib.strcpy (password_repeat, getpass ("Retype new password: "));       // prompt for and get repeated new password
         if (!dylib.strcmp (password, password_repeat)) {                         // confirm they are the same
            cipher_vic_encrypt (password, cipher, sizeof (cipher));               // generate a cipher for the new password
            if (dylib.strcmp (cipher, cipher_check)) {                            // confirm the password has been changed
               r = passwd_update_cipher (username, cipher);                       // update the passwd file with new password cipher
               if (r) {                                                           // confirm nothing went wrong updating the passwd file
                  dylib.fputs ("Password file could not be updated\n", stdout);   // output an error message
               }
            } else {
               dylib.fputs ("The password has not been changed.\n", stderr);      // output an error message
            }
         } else {
            dylib.fputs ("New and retyped passwords differ.\n", stderr);          // output an error message
         }
      } else {
         dylib.fputs ("Old password is incorrect.\n", stderr);                    // output an error message
      }
   } else {
      dylib.fputs ("passwd: unknown user\n", stderr);                             // output an error message
   }
     
   return 0;                                                                      // return the result
}
