#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>

#define CMD_REM   0
#define CMD_PRINT 1
#define CMD_FOR   2
#define CMD_END   3
#define CMD_NEXT  4
#define CMD_IF    5
#define CMD_GOTO  6
#define CMD_LET   7
#define CMD_EOF   8

#define PAGE_SIZE 4096

// program storage - multiple pages

#define PROG_PAGES_MAX 16
int prog_page_count;
int prog_page_cur;
char *prog;
char *prog_write_addr;
int prog_page_rem;

// variable definition storage - one page

#define VARTYPE_INT    0
#define VARTYPE_LONG   1
#define VARTYPE_DOUBLE 2
#define VARTYPE_STRING 3

typedef struct {
   char name[16];
   long type;
   long array_len;
   long page;
   void *addr;
} vardef_entry_t;

#define VARDEF_ENTRY_MAX (PAGE_SIZE / sizeof (vardef_entry_t))

typedef struct {
   vardef_entry_t entry[VARDEF_ENTRY_MAX];
   int count;
} vardef_t;

int var_const_count = 0;

vardef_t *vardef = NULL;

int var_page_count;
int var_page_cur;
char *var;
char *var_write_addr;
int var_page_rem;

typedef struct {
   long linenum;
   long cmd;
   long param_count;
   unsigned long param[16];
} statement_t;

// variable value storage - multiple pages

#define VARVAL_PAGES_MAX 16
int varval_page;
int varval_page_next_addr[VARVAL_PAGES_MAX];

void hexdump (const char *v, int len) {
   unsigned char *v1 = (unsigned char *) v;
   for (int i = 0; i < len; i++) {
      fprintf (stdout, "%02x", (unsigned int) *v1);
      v1++;
   }
   fprintf (stdout, "\n");
}

void variable_add (const char *name, int type, int array_len, unsigned long *page, unsigned long *addr) {
   if (!vardef) {
      vardef = malloc (PAGE_SIZE);
      memset (vardef, 0x00, sizeof (vardef_t));
   }
   
   int found = false;

   for (int i = 0; i < vardef->count; i++) {
      if (!strcmp (name, vardef->entry[i].name)) {
         *page = vardef->entry[i].page;
         *addr = (unsigned long) vardef->entry[i].addr;
         found = true;
         fprintf (stdout, "found variable %s at page %d %016lx\n", name, (int) vardef->entry[i].page, (unsigned long) vardef->entry[i].addr);
         break;
      }
   }

   if (!found) {
      strcpy (vardef->entry[vardef->count].name, name);
      vardef->entry[vardef->count].type      = type;
      vardef->entry[vardef->count].array_len = array_len;

      int var_len;
      switch (type) {
         case VARTYPE_INT:
            var_len = sizeof (int);
            break;
         case VARTYPE_LONG:
            var_len = sizeof (long);
            break;
         case VARTYPE_DOUBLE:
            var_len = sizeof (double);
            break;
         case VARTYPE_STRING:
            var_len = 256;
            break;
      }

      if (var_page_rem < var_len) {
         var            = malloc (PAGE_SIZE);
         var_write_addr = var;
         var_page_rem   = PAGE_SIZE;
      }
  
      vardef->entry[vardef->count].page = 0;
      vardef->entry[vardef->count].addr = var_write_addr;
      *page = vardef->entry[vardef->count].page;
      *addr = (unsigned long) vardef->entry[vardef->count].addr;
      fprintf (stdout, "added variable %s at page %d addr %016lx\n", name, (int) vardef->entry[vardef->count].page, (unsigned long) vardef->entry[vardef->count].addr);
      vardef->count++;
      var_write_addr += var_len;
      var_page_rem   -= var_len;

   }
}

void program_init () {
   prog_page_count = 0;
   prog_page_rem   = 0;
}

void statement_add (statement_t *statement) {
   int statement_len = 3 * sizeof (long) + statement->param_count * sizeof (long);
   if (prog_page_rem < statement_len) {
      prog            = malloc (PAGE_SIZE);
      prog_write_addr = prog;
      prog_page_rem   = PAGE_SIZE;
   }
   fprintf (stdout, "%ld %ld %ld %d\n", statement->linenum, statement->cmd, statement->param_count, statement_len);
   memcpy (prog_write_addr, statement, statement_len);
   hexdump (prog_write_addr, statement_len);
   hexdump ((char *) &statement->linenum, sizeof (long));
   hexdump ((char *) &statement->cmd, sizeof (long));
   hexdump ((char *) &statement->param_count, sizeof (long));
   for (int i = 0; i < statement->param_count; i++) {
      hexdump ((char *) &statement->param[i], sizeof (unsigned long));
   }
   prog_write_addr += statement_len;
   prog_page_rem -= statement_len;
}

int statement_process_print (statement_t *statement) {
   statement->cmd = CMD_PRINT;
   char *t = strtok (NULL, " \n");
   if (*t == '"') {
      t++;
      t[strcspn (t, "\"")] = 0x00;
      char name[16];
      sprintf (name, "CONST_%d", var_const_count++);
      variable_add (name, VARTYPE_STRING, 1, &statement->param[0], &statement->param[1]);
   } else {
      variable_add (t, VARTYPE_INT, 1, &statement->param[0], &statement->param[1]);
   }
   statement->param_count = 2;
   statement_add (statement);
   return 0;
}

int statement_process_for (statement_t *statement) {
   statement->cmd = CMD_FOR;
   char *t = strtok (NULL, " \n");    // variable
   variable_add (t, VARTYPE_INT, 1, &statement->param[0], &statement->param[1]);
   t = strtok (NULL, " \n");          // =
   t = strtok (NULL, " \n");          // initial value
   statement->param[2] = atoi (t);
   t = strtok (NULL, " \n");          // to
   t = strtok (NULL, " \n");          // final value
   statement->param[3] = atoi (t);
   t = strtok (NULL, " \n");          // step?
   if (t) {
      t = strtok (NULL, " \n");       // step value
      statement->param[4] = atoi (t);
   } else {
      statement->param[4] = 1;        // default step
   }
   statement->param_count = 5;
   statement_add (statement);
   return 0;
}

int statement_process_next (statement_t *statement) {
   statement->cmd = CMD_NEXT;
   char *t = strtok (NULL, " \n");    // variable
   variable_add (t, VARTYPE_INT, 1, &statement->param[0], &statement->param[1]);
   statement->param[2] = 0;           // page of corresponding for statement
   statement->param[3] = 0;           // address of corresponding for statement
   statement->param_count = 4;
   statement_add (statement);
   return 0;
}

int statement_process_end (statement_t *statement) {
   statement->cmd = CMD_END;
   statement->param_count = 0;
   statement_add (statement);
   return 0;
}

int statement_process_eof () {
   statement_t statement;
   statement.linenum = 0;
   statement.cmd = CMD_EOF;
   statement.param_count = 0;
   statement_add (&statement);
   return 0;
}

int statement_process_if (statement_t *statement) {
   statement->cmd = CMD_IF;
   statement->param_count = 0;
   statement_add (statement);
   return 0;
}

int statement_process_let (statement_t *statement) {
   statement->cmd = CMD_LET;
   char *t = strtok (NULL, " \n");    // variable
   variable_add (t, VARTYPE_INT, 1, &statement->param[0], &statement->param[1]);
   t = strtok (NULL, " \n");          // =
   t = strtok (NULL, " \n");          // value
   statement->param[2] = atoi (t);
   statement->param_count = 3;
   statement_add (statement);
   return 0;
}

int statement_process_goto (statement_t *statement) {
   statement->cmd = CMD_GOTO;
   statement->param_count = 0;
   statement_add (statement);
   return 0;
}

int statement_process_rem (statement_t *statement) {
   statement->cmd = CMD_REM;
   statement->param_count = 0;
   statement_add (statement);
   return 0;
}

int program_load (const char *path) {
   int r = 1;

   FILE *f;

   if ((f = fopen (path, "r"))) {

      program_init ();

      char s[256];
      char *t;
      statement_t statement;
      int e;

      while (fgets (s, sizeof (s), f)) {

         t = strtok (s, " \n");
         statement.linenum = atoi (t);
         if (statement.linenum) {
            t = strtok (NULL, " \n");
         }

         e = 1;
         if (!strcasecmp (t, "PRINT")) {
            e = statement_process_print (&statement);
         } else if (!strcasecmp (t, "FOR")) {
            e = statement_process_for (&statement);
         } else if (!strcasecmp (t, "END")) {
            e = statement_process_end (&statement);
         } else if (!strcasecmp (t, "REM")) {
            e = statement_process_rem (&statement);
         } else if (!strcasecmp (t, "IF")) {
            e = statement_process_if (&statement);
         } else if (!strcasecmp (t, "GOTO")) {
            e = statement_process_goto (&statement);
         } else if (!strcasecmp (t, "NEXT")) {
            e = statement_process_next (&statement);
         } else if (!strcasecmp (t, "LET")) {
            e = statement_process_let (&statement);
         } else {
            fprintf (stderr, "syntax error: %s\n", t);
         }
      }
      statement_process_eof ();
      fclose (f);
      r = 0;
   }

fprintf (stdout, "done reading\n");

   return r;
}

int program_exec () {
   int r = 0;
   char *prog_ctr;
   int statement_len;
   int i;

   prog_ctr = prog;
   statement_t *statement = (statement_t *) prog_ctr;

   while (1) {
      statement_len = 3 * sizeof (long) + statement->param_count * sizeof (long);
      fprintf (stdout, "%ld %ld %d ", statement->linenum, statement->cmd, statement_len);
      for (i = 0; i < statement->param_count; i++) {
         fprintf (stdout, "%lx ", statement->param[i]);
      }
      fprintf (stdout, "\n");

      if (statement->cmd == CMD_EOF) {
         break;
      }
      prog_ctr += statement_len;
      statement = (statement_t *) prog_ctr;
   }
   return r;
}

void usage_print () {
   fputs ("usage: ubasic program_path\n", stderr);
}

int main (int argc, char *argv[]) {

   int r;

   if (argc != 2) {
      usage_print ();
      exit (1);
   }

   r = program_load (argv[1]);
   if (r) {
      fprintf (stderr, "cannot open %s\n", argv[1]);
      exit (1);
   }

   r = program_exec ();

   return 0;
}
