/* rm.c
   This program removes directory entries.
   Usage: rm file_1 [... file_n]
   change log:
   08/15/2023 initial version
   08/15/2023 added error cases
   08/21/2023 replaced usage of fprintf with dylib.fputs
   12/10/2023 added reference to conversion.h
   12/13/2023 added comments
              added getopt logic
   02/27/2024 updated to use dylib
   01/20/2025 updated fputs to dylib
   06/03/2025 added recursive removal and verbosity
   06/08/2025 added call to remove the batch file
   06/09/2025 corrected printing of individual files when verbose
   06/10/2025 moved temp file to /tmp
   06/16/2025 updated exit reference
   09/18/2025 added error message for non-existent file/directory parameter
*/ 

#include <stdlib.h>
#include <unistd.h>
#include <dylib.h>
#include <sys/stat.h>
#include <batch_private.h>
#include <utility_private.h>

// prints usage info
void print_usage () {
   dylib.fputs ("usage: rm [-i] [-r] [-v] [file ...]\n", stdout);
}

#define BATCH_NAME "/tmp/t_rm"

int main (int argc, char *argv[]) {

   bool error       = false;                                        // indicates whether there was an error in processing parameters
   bool recursive   = false;                                        // recursive removal
   bool verbose     = false;                                        // verbosity of operations
   bool interactive = false;
   bool debug       = false;

   // capture the options
   int opt;
   while ((opt = getopt (argc, argv, "irvd")) != -1) {
      switch (opt) {
         case 'd':                                                  // handle debug
            debug       = true;
            break;
         case 'i':                                                  // handle recursive removal
            interactive = true;
            break;
         case 'r':                                                  // handle recursive removal
            recursive   = true;
            break;
         case 'v':                                                  // handle verbosity
            verbose     = true;
            break;
         case '?':                                                  // the error cases
         default:
            error = true;
            break;
      }
   }

   // adjust argc and argv removing the processed options
   argc -= optind;
   argv += optind;

   if (error || !argc) {                                            // exit if an error occured while processing options or no file
      print_usage ();                                               // arguments were provided
      exit (0);
   }


   struct stat statbuf;
   int r = 0;
   batch_t batch;

   for (int i = 0; i < argc; i++) {                                 // loop through all the parameters, which should be files
      r = stat (argv[i], &statbuf);                                 // stat the file
      if (!r) {                                                     // if exists
         if (S_ISREG (statbuf.st_mode)) {                           // if is a regular file
            r = remove (argv[i]);                                   // remove the file
         } else if (S_ISDIR (statbuf.st_mode)) {                    // if is a directory
            if (recursive) {                                        // if is recursive removal
               batch = batch_create (BATCH_NAME, false, verbose, debug);   // create a batch file
               if (!interactive || file_examine_directory_query (argv[i])) {
                  r = dir_remove_contents_recursive (&batch, argv[i], interactive); // capture what needs to be deleted
               }
               batch_close (&batch);                                // close the batch file
               batch_execute (&batch);                              // execute the batch file
               batch_remove (&batch);            
               remove (argv[i]);                                    // remove the directory
            } else {                                                // print a diagnostic message indicating the file is a dir
               dylib.fputs ("rm: ", stderr);                        // when recursive option was not specified
               dylib.fputs (argv[i], stderr);
               dylib.fputs (": is a directory\n", stderr);
               r = 1;
            }
         }
         if (!r && verbose) {
            dylib.fputs (argv[i], stdout);
            dylib.fputs ("\n", stdout);
         }
      } else {
         dylib.fputs ("rm: ", stderr);
         dylib.fputs (argv[i], stderr);
         dylib.fputs (": No such file or directory\n", stderr);
      }
   }

   return r;                                                        // return result
}
