/* sh.c
   This program provides a shell command line interface to execute commands.
   Commands are of the form: cmd1 [arg1 [... argn]]
   Commands may be chained sequentially, separated by a semi-colon: cmd1 ; cmd2, or
   via a pipe where the stdout of one cmd is piped to the stdin of a subsequent command: cmd1 | cmd2.
   Command stdout can also be redirected to a file: cmd1 > file.
   Commands and arguments may be quoted to allow for inclusion of spaces and other special characters, other than other quotes.
   Examples sequences:
      cmd1
      cmd1 p1 p2
      cmd1 p1 p2 | cmd
      cmd1 p1 p2 > f1
      cmd1 ; cmd2
      cmd1 "p1" p2
   Example commands:
      ls                                                       lists the contents of the current working directory
      ls -l /WDS1                                              lists the directory contents of WDS1.
      df /WDS1 /DSK1 /DSK2                                     displays disk free stats for each of the specified file systems
      cat John1                                                displays the contents of the file John1, in the current working dir
      cat /WDS1/John1                                          displays the contents of the file WDS1.John1
      more /WDS1/KJV_1611_4                                    displays the contents of the KJV Bible one page at a time
      cat /WDS1/John1 | grep -i -w word | tee /WDS1/John1_word reads the file WDS1.John1 and pipes output to grep which will
                                                               search for all lines for text with 'word' as whole words and 
                                                               case-insensitive. It will then pipe the result to tee, which 
                                                               will both write to the file WDS1.John1_word and display to stdout.
      grep -i -w word /WDS1/KJV_1611_4 > /WDS1/KJV_word        searches the KJV Bible for the text 'word' as whole words and
                                                               case-insensitive, redirecting the output to the file WDS1.KJV_word.
      grep -i "word was god" /WDS1/John1 | more                searches the file WDS1.John1 for the string "word was god" (without
                                                               quotes) and pipes to the more command.
   change log:
   06/01/2023 initial version
   07/05/2023 updated aliasing message output
   07/21/2023 added test for file dsr_status to ensure that all commands are program files
   07/22/2023 added test for null command, which caused the call to file_get_dsr_status to get upset and write all over vdp memory 
              (yikes)
   07/27/2023 added exit alias
   08/05/2023 removed commented out method get_unaliased_cmd, which was previously movied to common code that also supports execv.
   08/05/2023 added support for quoted commands and arguments. Since the TI file systems don't support spaces in the filenames,
              this feature is less useful yet added for completeness. It is useful for grep's search string.
   08/05/2023 renamed to sh.c
   08/05/2023 added cwd support
   08/14/2023 updated for unix paths
   08/18/2023 updated to use system_private.h and print real and available memory
   08/19/2023 updated to gather full path to command, whether aliased or not
   08/25/2023 replaced use of sprintf with strcpy and strcat to reduce memory footprint
   09/02/2023 replaced hard-coded path with get_root_fs_
   09/04/2023 corrected logic in init_cwd so that it gracefully fails when no cwd is part of the process info nor in the proc/cwd
              file
   09/08/2023 updated to search for UNIX99 INITD ROM
   09/24/2023 added logic to detect and process redirect-append as mode a, and indicate all pipes as mode w
   11/02/2023 updated for system_private.h name changes
   11/03/2023 migrated most of the shell commanding functionality to the system command
              replaced direct access to private cache values with public calls to cache methods
              replaced access to the unistd private API to the public API
   11/10/2023 removed splash screen and testing of whether INITD is present
   11/15/2023 updated comments
   11/21/2023 updated to use FILENAME_MAX
   02/27/2024 added use of dylib
   03/01/2024 additional ue of dylib
   03/06/2024 modified to loop infinitely at the prompt and exit indirectly through a successfully system() call
   03/07/2024 updated to use dylib's getcwd
   01/18/2025 updated description of prompt generated
   01/20/2025 moved fputs to dylib
   06/08/2025 handle \n returned by fgets
   06/09/2025 handle ctrl-d / EOF on stdin
              set to ignore SIGINT
   06/24/2025 changed error message from syntax error to just error
*/

#include <string.h>
#include <stdbool.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <soundqueue.h>
#include <dylib.h>
#include <signal.h>

// capture the cwd
char *init_cwd () {
   static char cwd[FILENAME_MAX];
   char *p;
   p = dylib.getcwd (cwd, sizeof (cwd));     // get the cwd. Unlikely set as part of the process info since this is the shell.
   if (!p) {                                 // if all the above failed, then the cwd isn't available at all
      dylib.strcpy (cwd, "no cwd");          // so save this. this is a sorry state
   }
   return cwd;
}

// determine if string has any meaningful text
bool has_text (const char *s) {
   bool r = false;
   while (*s) {
      if (*s > 32) {
         r = true;
         break;
      }
      s++;
   }
   return r;
}

// the main method
int main (int argc, char *argv[]) {

   char s[256];

   signal (SIGINT, SIG_IGN);

   int r = 1;

   char *cwd = init_cwd ();                            // initialize cwd, which will not have been set by init cartridge-based daemon

   int cont = 1;
   while (cont) {                                      // loop until commands have been successfully added
                                                       // display prompt of the form: [marko@napili /home/marko]$
      dylib.fputs ("[", stdout);                       // opening prompt
      dylib.fputs (getlogin (), stdout);               // write the user name
      dylib.fputs (" ", stdout);                       // write a space
      dylib.fputs (cwd, stdout);                       // write the current working directory
      dylib.fputs ("]$ ", stdout);                     // finish the prompt
      if (dylib.fgets (s, sizeof (s), stdin)) {        // read commands from stdin
         s[dylib.strcspn (s, "\r\n")] = 0x00;
         if (has_text (s)) {                           // only call the system command if there's something to process
            r = system (s);                            // process commands from the user
            if (r) {                                   // notify the user on error, 
               soundqueue_honk ();                     // with a honk
               dylib.fputs ("error\n", stdout);        // and some text
            }
         }
      } else {
         cont = false;
      }
   }
 
   // this section will only be reached on closure of stdin (EOF / ctrl-d)
   dylib.fputs ("\n", stdout);

   return 0;
}
