/* sort.c
   This program sorts the input of a file.
   Usage: sort [-f] [-u] [-o output] [-r] [file ...]
   Options:
   -f sort case insensitive
   -u sort outputing only unique values
   -o output to a specified file
   -r output in reverse order
   change log:
   12/11/2023 initial version
   12/12/2023 added output in reverse order
              added case insensitive sorting and removal of lexically equal strings when sorting for unique items
   12/13/2023 added comments
              modified to handle sort order by multiplying the string comparison functions by 1 or -1
   02/27/2024 updated to use dylib
   03/01/2024 more extensive use of dylib
   01/20/2025 updated fputs to use dylib
   06/08/2025 handle fgets / strcspn
   06/16/2025 updated exit reference
*/

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdbool.h>
#include <stdio.h>
#include <dylib.h>
#include <stdlib.h>

int sort_order = 1;

// prints usage info
void print_usage () {
   dylib.fputs ("usage: sort [-f] [-u] [-o output] [-r] [file ...]\n", stdout);
}

// compares two strings case-sensitive for qsort
int local_strcmp (const void *a, const void *b) { 
    return dylib.strcmp (*(const char**)a, *(const char**)b) * sort_order; 
} 
 
// compares two strings case-insensitive for qsort
int local_strcasecmp (const void *a, const void *b) {
    return strcasecmp (*(const char**)a, *(const char**)b) * sort_order;
}

// sorts an array of strings
void sort (char* arr[], int n, bool sort_case_sensitive) { 
   if (sort_case_sensitive) {
      qsort (arr, n, sizeof(const char*), local_strcmp);     // sort case-sensitively
   } else {
      qsort (arr, n, sizeof(const char*), local_strcasecmp); // sort case-insensitively
   }
} 

// sorts one file
void sort_file (FILE *fin, FILE *fout, bool sort_unique, bool sort_case_sensitive) {
   char s[256];                                   // input buffer
   char *t[512];
   int n = 0;   
  
   while (dylib.fgets (s, sizeof (s), fin)) {     // read from the file
      t[n] = malloc (dylib.strlen (s) + 1);       // allocate storage for this line of text
      if (t[n]) {                                 // make sure the malloc succeeded
         dylib.strcpy (t[n], s);                  // copy the string to the array
         n++;                                     // increment the array count
      } else {                                    // handle failed malloc
         dylib.fputs ("sort: out of memory\n", stderr); // output an error message
         exit (0);                          // exit the program
      }
   }
   
   sort (t, n, sort_case_sensitive);              // sort the array

   int i, j;
   int inc = 1;

   // handle sorting for unique items
   if (sort_unique) {                             // handle sorting for unique items
      for (i = 0; i < n - 1; i += inc) {          // loop through all items
         inc = 1;                                 // set the loop incrementer to move to the next item by default
         if ( (sort_case_sensitive &&             // assess if this and the next string are identical
               !dylib.strcmp (t[i], t[i + 1])) ||
              (!sort_case_sensitive &&            // assess if this and the next string are lexically identical
               !strcasecmp (t[i], t[i + 1])) ) {   
            free (t[i + 1]);                      // they are. free this string
            for (j = i + 1; j < n - 1; j++) {     // move all remaining strings in the array up one
               t[j] = t[j + 1];
            }
            n--;                                  // decrement the array count
            inc = 0;                              // set the loop incrementer to zero in case the subsequent value is the same as
         }                                        // the current
      }
   }

   // output the sorted list
   for (i = 0; i < n; i++) {                      // loop through all items in the array
      dylib.fputs (t[i], fout);                   // print the item
      free (t[i]);                                // free the array element
   }
}

// the sort program
int main (int argc, char *argv[]) {

   char outfile[FILENAME_MAX];                          // optional outfile name
   bool use_outfile         = false;                    // indicates whether to use the outfile
   bool sort_unique         = false;                    // indicates whether to sort and output only unique values
   bool sort_case_sensitive = true;                     // indicates whether the sort should be case senitive or insensitive
   bool error               = false;                    // indicates whether there was an error in processing parameters

   // capture the options
   int opt;
   while ((opt = getopt (argc, argv, "fo:ru")) != -1) {
      switch (opt) {
         case 'f':                                      // handle case insensitivity
            sort_case_sensitive = false;
            break;
         case 'o':                                      // handle capturing an output file
            if (dylib.strlen (optarg)) {
               dylib.strcpy (outfile, optarg);
               use_outfile = true;
            } else {
               error = true;
            }
            break; 
         case 'r':                                      // handle sorting in reverse order
            sort_order = -1;             
            break;
         case 'u':                                      // handle sorting for unique records
            sort_unique = true;
            break;
         case '?':                                      // the error cases
         default:
            error = true;
            break;
      }
   }

   // exit if there was an error while processing options
   if (error) {
      print_usage ();
      exit (0);
   }

   // adjust argc and argv removing the processed options
   argc -= optind; 
   argv += optind; 

   // the input and output file streams
   FILE *fout, *fin;
   
   // set the output file
   if (use_outfile) {                                         // use the specified outfile
      fout = dylib.fopen (outfile, "w");
      if (!fout) {                                            // test for an open file, error out if not open
         dylib.fputs ("sort: No such file or directory\n", stderr);
         exit (0);
      }
   } else {
      fout = stdout;                                          // set the output to stdout
   }

   if (argc) {
      // loop through all files, confirming all exist before processing
      int i;
      for (i = 0; i < argc; i++) {
         fin = dylib.fopen (argv[i], "r");
         if (!fin) {
            dylib.fputs ("sort: No such file or directory\n", stderr);
            exit (0);
         }
         dylib.fclose (fin);
      }

      // loop through all files, processing them
      for (i = 0; i < argc; i++) {
         // open the file
         fin = dylib.fopen (argv[i], "r");
         // confirm the file is open -- this should always work since all files were tested above
         if (fin) {
            sort_file (fin, fout, sort_unique, sort_case_sensitive);
            dylib.fclose (fin);
         } else {
            dylib.fputs ("sort: No such file or directory\n", stderr);
            exit (0);
         }
      }
   } else {
      // no files were specified to sort stdin
      sort_file (stdin, fout, sort_unique, sort_case_sensitive);
   }

   // return the result
   return 0;
}
