/* tail.c
   This program display the last part of a file.
   Usage: tail [-n count] [-q] [file ...]
   Options:
   -n number of lines
   -q suppress printing of filename headers when multiple files are printed
   change log:
   08/20/2023 initial version
   08/20/2023 reduced max buffered lines from 128 to 64
   08/20/2023 changed use of fprintf for dylib.fputs to reduce memory footprint
   12/10/2023 removed reference to string_ext.h
   02/27/2024 added use of dylib
   01/20/2025 updated fputs to use dylib
   06/08/2025 updated for fgets / strcspn
   06/16/2025 updated exit reference
*/

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdbool.h>
#include <stdio.h>
#include <assert.h>
#include <dylib.h>

#define BUFFER_MAX_LINES 64                      // max number of lines to be buffered

// prints program usage info
void print_usage () {
   dylib.fputs ("usage: tail [-n lines] [-q] [file ...]\n", stdout);
}

// processes one file
void process_file (FILE *f, int max_count, bool print_filename, const char *filename) {
   static int fc = 0;                            // file count
   char s[256];                                  // input buffer

   char *p[BUFFER_MAX_LINES];                    // set up the line buffer
   int np = 0;
   for (int i = 0; i < BUFFER_MAX_LINES; i++) {
      p[i] = NULL;
   }

   if (print_filename) {                         // handle printing file headers
      if (fc) {                                  // handle space between files
         dylib.fputs ("\n", stdout);
      }
      fc++;                                      // increment file count
      dylib.fputs ("==> ", stdout);              // output the file header
      dylib.fputs (filename, stdout);
      dylib.fputs (" <==\n", stdout);
   }
      
   while (dylib.fgets (s, sizeof (s), f)) {      // process all lines

      if (np >= max_count) {                     // reset rolling buffer to first position if last has been exceeded
         np = 0;
      }
      if (p[np]) {                               // roll over the older part of the buffer if data exists there
         free (p[np]);                           // free the data
         p[np] = NULL;
      }
      p[np] = malloc (dylib.strlen (s) + 1);     // allocate memory for the new data
      assert (p[np]);
      dylib.strcpy (p[np], s);                   // copy the data
      np++;                                      // increment the counter
   }

   for (int i = 0; i < max_count; i++) {         // write out the buffer
      if (np >= max_count) {                     // reset the rolling buffer to the first position if the last has been exceeded
         np = 0;
      }
      if (p[np]) {                               // output the data if it exists
         dylib.fputs (p[np], stdout);
         free (p[np]);                           // free the data
         p[np] = NULL;
      }
      np++;
   }
}

// the tail program
int main (int argc, char *argv[]) {

   int max_count              = 10;
   bool error                 = false;
   bool pr_headers_mult_files = true;

   // capture the options
   int opt;
   while ((opt = getopt (argc, argv, "n:q")) != -1) {
      switch (opt) {                            
         case 'n':                             
            max_count = atoi (optarg);
            if ((max_count <= 0) || (max_count > BUFFER_MAX_LINES)) {
               dylib.fputs ("tail: illegal line count\n", stderr);
               error = true;
            }
            break; 
         case 'q':
            pr_headers_mult_files = false;
            break;
         case '?':
         default:
            error = true;
            break;
      }
   }

   // exit if there was an error while processing options
   if (error) {
      print_usage ();
      exit (0);
   }

   // adjust argc and argv removing the processed options
   argc -= optind; 
   argv += optind; 

   if (argc) {
      // loop through all files
      for (int i = 0; i < argc; i++) {
         // open the file
         FILE *f = dylib.fopen (argv[i], "r");
         // confirm the file is open
         if (f) {
            process_file (f, max_count, pr_headers_mult_files && (argc > 1), argv[i]);
            dylib.fclose (f);
         } else {
            // file failed to open
            dylib.fputs ("tail: ", stderr);
            dylib.fputs (argv[i], stderr);
            dylib.fputs (": no such file or directory\n", stderr);
         }
      }
   } else {
      process_file (stdin, max_count, false, NULL);
   }

   return 0;
}
