/* tmatrix.c
   This program implements a simple depiction of the Matrix construct viewer on
   board the ship Nebucanezzer from the movie 'The Matrix'.
   Usage: tmatrix
   Options:
   None
   Change Log:
   11/30/2024 initial version
   12/11/2024 updated for spec change on console_font_get_text
   01/20/2025 moved console_write_raw to dylib
   01/22/2025 performance improvements
   02/28/2025 updated for removal of 80x30 mode
   05/30/2025 updated to unblank the screen (continuously)
   06/05/2025 handle signals
   06/07/2025 moved calls to dylib
   07/05/2025 dylib adjustments
*/

#include <console.h>
#include <console_private.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <dylib.h>

// screen dimensions
#define HEIGHT 24
#define WIDTH  32

// visual and background grid
char vgrid[HEIGHT][WIDTH];
char bgrid[HEIGHT][WIDTH];

// control values
typedef struct {
   int start;
   int stop;
   int speed;
   int update;
} control_t;

control_t control[WIDTH];

// random values
#define CONTROL_RAND_NUM 80
const int control_rand[CONTROL_RAND_NUM] = {
   7, 1, 9, 10, 2, 8, 8, 14, 3, 13, 8, 5, 12, 2, 3, 7, 7, 1, 8, 4, 15, 1, 13, 5, 8, 
   5, 11, 6, 8, 7, 9, 2, 8, 3, 2, 1, 3, 13, 3, 5, 15, 0, 1, 8, 13, 1, 12, 11, 2, 5, 
   9, 1, 4, 8, 4, 2, 12, 4, 10, 7, 10, 10, 5, 8, 9, 5, 1, 0, 4, 5, 10, 5, 11, 13, 9,
   6, 7, 8, 1, 5
};
int control_ri = 0;

// random values for character generation 0-25
#define CHAR_RAND_NUM 80
const int char_rand[CHAR_RAND_NUM] = {
   11, 17, 5, 10, 16, 24, 20, 16, 5, 9, 10, 23, 24, 16, 21, 13, 17, 19, 24, 18, 5, 
   17, 25, 17, 12, 25, 11, 24, 6, 5, 21, 4, 20, 11, 16, 5, 15, 3, 1, 7, 11, 16, 3, 
   16, 17, 17, 2, 17, 22, 3, 13, 23, 4, 20, 14, 16, 16, 4, 10, 11, 0, 8, 19, 6, 3, 
   15, 7, 2, 18, 15, 8, 9, 19, 7, 1, 18, 5, 24, 5, 21
};
int char_ri = 0;

// returns random in the range 0 to 15
int control_rand_get () {
   int x = control_rand[control_ri];
   control_ri++;
   if (control_ri >= CONTROL_RAND_NUM) {
      control_ri = 0;
   }
   return x;
}

// returns a random value between 0 and 25
int char_rand_get () {
   int x = char_rand[char_ri];
   char_ri++;
   if (char_ri >= CHAR_RAND_NUM - 1) {
      char_ri = 0;
   }
   return x;
}

// initializes the background grid for one row
void bgrid_init_col (int c) {
   int r;
   for (r = 0; r < HEIGHT; r++) {
      bgrid[r][c] = 33 + 32 * (char_rand_get () % 2) + char_rand_get ();
   }
}

// initializes the background grid
void bgrid_init () {
   int c;
   for (c = 0; c < WIDTH; c++) {
      bgrid_init_col (c);
   }
}

// updates the background grid
void bgrid_update () {
   int c;
   for (c = 0; c < WIDTH; c++) {
      control[c].update++;
      if (control[c].update == control[c].speed) {
         control[c].update = 0;
         control[c].start++;
         control[c].stop++;
         if (control[c].start >= HEIGHT) {
            control[c].stop   = -(control_rand_get () / 3);
            control[c].start  = control[c].stop - control_rand_get () * 2;
            control[c].speed  = 1 + control_rand_get () % 3;
            control[c].update = 0;

            bgrid_init_col (c);
         }
      }
   }
}

// initializes the control structures
void control_init () {
   int c;
   for (c = 0; c < WIDTH; c++) {
      control[c].stop   = -(control_rand_get () / 3);
      control[c].start  = control[c].stop - control_rand_get () * 2;
      control[c].speed  = 1 + control_rand_get () % 3;
      control[c].update = 0;
   }
}

// initializes the visual grid screen attributes
void vgrid_init () {
   int g;
   console_text_set_background_color (COLOR_BLACK);
   for (g = 4; g < 8; g++) {
      console_standard_set_char_group_color (g, COLOR_LTGREEN, COLOR_TRANS);
   }
   for (g = 8; g < 12; g++) {
      console_standard_set_char_group_color (g, COLOR_DKGREEN, COLOR_TRANS);
   }
   for (g = 12; g < 16; g++) {
      console_standard_set_char_group_color (g, COLOR_WHITE, COLOR_TRANS);
   }

/*
   int c;
   char cdef[18];
   for (c = 65; c < 96; c++) {
      console_font_get_text (c, cdef);
      console_font_set_text (c - 32, cdef);
      console_font_set_text (c + 32, cdef);
   }
*/
}

// generates the visual grid based on the background grid and the control 
// structures
void vgrid_generate () {
   int r, c;
   for (r = 0; r < HEIGHT; r++) {
      for (c = 0; c < WIDTH; c++) {
         if (control[c].start <= r && control[c].stop >= r) {
            if (control[c].stop == r) {
               if (bgrid[r][c] >= 65) {
                  vgrid[r][c] = bgrid[r][c] + 32;
               } else {
                  vgrid[r][c] = bgrid[r][c] + 64;
               }
            } else {
               vgrid[r][c] = bgrid[r][c];
            }
         } else {
            vgrid[r][c] = ' ';
         }
      }
   }
}

// display the visual grid
void vgrid_display () {
   int r;
   for (r = 0; r < HEIGHT; r++) {
      console_write_raw (r, 0, vgrid[r], WIDTH);
   }
}

// screen mode to which to restore
int restore_mode;
int restore_rows;

// restore graphics at termination
void prog_term () {

   // restore the screen
   if (restore_mode != DISPLAY_MODE_STANDARD ||
       restore_rows != DISPLAY_ROWS_24) {
      console_display_set_mode (restore_mode, restore_rows);
   }
   console_cls ();
   console_standard_set_default_color ();
   console_fonts_load_std ();
}

// the main method
int main (int argc, char *argv[]) {

   // get the display mode to which to restore
   console_display_get_mode (&restore_mode, &restore_rows);
   
   // change screen modes if required
   if (restore_mode != DISPLAY_MODE_STANDARD ||
       restore_rows != DISPLAY_ROWS_24) {
      console_display_set_mode (DISPLAY_MODE_STANDARD, DISPLAY_ROWS_24);
   }

   // set up atexit
   dylib.atexit (prog_term);

   // initialize the control, background grid and visual grid
   control_init ();
   bgrid_init ();
   vgrid_init ();

   // loop forever or until a key is pressed
   int k;
   while (1) {
      // read the keyboard and exit if 'x' is pressed
      k = console_get_key ();
      if (k != 0xff) {
         break;
      }

      // generate the visual grid
      vgrid_generate ();

      // display the visual grid
      vgrid_display ();

      // update the background grid
      bgrid_update ();

      console_display_unblank ();
   }

   return 0;
}
