/* uname.c
   This program displays information about the system.
   Usage: uname [-amnoprsv]
     -a      Behave as though the options -m, -n, -r, -s, and -v were specified.
     -m      Write the type of the current hardware platform to standard output.  (make(1) uses it to set the MACHINE variable.)
     -n      Write the name of the system to standard output.
     -o      This is a synonym for the -s option, for compatibility with other systems.
     -p      Write the type of the machine processor architecture to standard output.  (make(1) uses it to set the MACHINE_ARCH
             variable.)
     -r      Write the current release level of the operating system to standard output.
     -s      Write the name of the operating system implementation to standard output.
     -v      Write the version level of this release of the operating system to standard output.
   change log:
   08/18/2023 initial version
   08/18/2023 updated to use values from system_private.h
   11/02/2023 updated to use utsname.h methods rather that directly using values from system_private.h
   02/27/2024 updated to use dylib
   01/20/2025 moved fputs to dylib
   06/16/2025 updated exit reference
*/ 

#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <stdio.h>
#include <utsname.h>
#include <dylib.h>

void print_usage () {
   dylib.fputs ("usage: uname [-amnoprsv]\n", stdout);
}

int main (int argc, char *argv[]) {

   // capture the options
   bool m = false;
   bool n = false;
   bool p = false;
   bool r = false;
   bool s = false;
   bool v = false;

   bool error = false;

   int opt;
   while (((opt = getopt (argc, argv, "amnoprsv")) != -1) && !error) { // loop through all options
      switch (opt) {                                     
         case 'a':                                       
            m = true;
            n = true;
            r = true;
            s = true;
            v = true;
            break;
         case 'm':                                        
            m = true;
            break;
         case 'n':                                         
            n = true;
            break;
         case 'p':
            p = true;
            break; 
         case 'r':
            r = true;
            break;
         case 'o':
         case 's':
            s = true;
            break;
         case 'v':
            v = true;
            break;
         case '?':
         default:
            error = true;
            break;
      }
   } 

   if (error) {
      print_usage ();
      exit (0);
   }

   // adjust argc and argv removing the processed options
   argc -= optind;  
   argv += optind;

   if (!(m || n || p || s || v)) {
      s = true;
   }
 
   struct utsname uinfo;
   uname (&uinfo);

   if (s) {
      dylib.fputs (uinfo.sysname, stdout);
      dylib.fputs (" ", stdout);
   }

   if (n) {
      dylib.fputs (uinfo.nodename, stdout);
      dylib.fputs (" ", stdout);
   }

   if (v) {
      dylib.fputs (uinfo.version, stdout);
      dylib.fputs (" ", stdout);
   }

   if (r) {
      dylib.fputs (uinfo.release, stdout);
      dylib.fputs (" ", stdout);
   }

   if (p) {
      dylib.fputs (uinfo.proc_arch, stdout);
      dylib.fputs (" ", stdout);
   }

   if (m) {
      dylib.fputs (uinfo.machine, stdout);
      dylib.fputs (" ", stdout);
   }

   dylib.fputs ("\n", stdout);

   return 0;
}
