/* MIDI to TMS9919 Sound Byte Generator

   This program is intended to provide a simple conversion of relatively basic midi files to the three
   voices of the 9919. 

   - Parameters
     - midi filename
     - b : insert buffer fill notes (0 duration notes) TBA FEATURE
     - c : insert comments from midi translation

   Limitations:
   - No support for more than three voices. If more than 3 are encountered the program will warn and
     ignore the extra. The resulting sound product will undoubtedly be less than desirable.
   - Little attempt is made to reduce the overall size of the tones
     - All voices are turned off at the beginning of the second and subsequent notes
     - Voices frequencies are set for each midi note on request, and corresponding volume
     - Midinotes are mapped to voices as found in the incoming data, no attempt is made to remap
       the midinotes to the previously used voice, which may still have the frequency set for that note.
     Although this program produces correct sound it wastes space and that can leaded to reduced overall
     sound quality if sound buffer management and timing is less than perfect.
   - The TI99's 9919, at least as emulated in MAME, does not acheive Interrupt Service Request handling 
     at the promised 1/60s (NTSC) or 1/50s (PAL). The reason is unclear. ISR_performance_factor is set
     to 0.667 to address the difference in duration. This is only an estimate but seems fairly close.
     The resolution of tone duration is whatever the ISR rate actual is, nominally being 1/60 or 16.7 ms.
     9919 tone duration = duration * ISR rate.

   The 9919 terminology in context of midi can be a little challenging. A note in 9919 venacular is 
   simply a size and corresponding set of commands for frequencies, noises and volumes and duration. A
   note in midi is more precisely a single note for a single instrument. The TMS9919 is essentially a
   single instrument, similar to an organ.

   Dependencies:
   - This program uses mididump from midilib (https://github.com/MarquisdeGeek/midilib). Download and
     make sure at least the program mididump is in the path.

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

// true and false
#define TRUE  1
#define FALSE 0

// undefined
#define UNDEFINED -1

// voices on and off
#define VOICE_OFF FALSE
#define VOICE_ON  TRUE

// number of voices and noises
#define NUM_VOICES 3
#define NUM_NOISES 1

// map the TMS9919 sound generator voices to 0-2 and noise to 3
const int tone_freq_cmd[NUM_VOICES + NUM_NOISES] = {0x80, 0xA0, 0xC0, 0xE0};
const int tone_vol_cmd[NUM_VOICES + NUM_NOISES]  = {0x90, 0xB0, 0xD0, 0xF0};

// Supposedly the 9919 is clocked such that notes are played for count * 1/60s NTSB or 1/50s PAL,
// but at least in MAME this rate is off. Set accordingly if you experience something different.
const double ISR_performance_factor = 0.667;

// table to translate midi notes to frequencies
double midinote2freq[128];

// loads the midi notes to frequencies table
void init_midinote2freq () {

   // note: although midinote2freq could have been populated as a const array, setting these values
   // individually here provides a nice cross reference matrix.

   // load the values
   midinote2freq[0] = 8.18;
   midinote2freq[1] = 8.66;
   midinote2freq[2] = 9.18;
   midinote2freq[3] = 9.72;
   midinote2freq[4] = 10.30;
   midinote2freq[5] = 10.91;
   midinote2freq[6] = 11.56;
   midinote2freq[7] = 12.25;
   midinote2freq[8] = 12.98;
   midinote2freq[9] = 13.75;
   midinote2freq[10] = 14.57;
   midinote2freq[11] = 15.43;
   midinote2freq[12] = 16.35;
   midinote2freq[13] = 17.32;
   midinote2freq[14] = 18.35;
   midinote2freq[15] = 19.45;
   midinote2freq[16] = 20.60;
   midinote2freq[17] = 21.83;
   midinote2freq[18] = 23.12;
   midinote2freq[19] = 24.50;
   midinote2freq[20] = 25.96;
   midinote2freq[21] = 27.50;
   midinote2freq[22] = 29.14;
   midinote2freq[23] = 30.87;
   midinote2freq[24] = 32.70;
   midinote2freq[25] = 34.65;
   midinote2freq[26] = 36.71;
   midinote2freq[27] = 38.89;
   midinote2freq[28] = 41.20;
   midinote2freq[29] = 43.65;
   midinote2freq[30] = 46.25;
   midinote2freq[31] = 49.00;
   midinote2freq[32] = 51.91;
   midinote2freq[33] = 55.00;
   midinote2freq[34] = 58.27;
   midinote2freq[35] = 61.74;
   midinote2freq[36] = 65.41;
   midinote2freq[37] = 69.30;
   midinote2freq[38] = 73.42;
   midinote2freq[39] = 77.78;
   midinote2freq[40] = 82.41;
   midinote2freq[41] = 87.31;
   midinote2freq[42] = 92.50;
   midinote2freq[43] = 98.00;
   midinote2freq[44] = 103.83;
   midinote2freq[45] = 110.00;
   midinote2freq[46] = 116.54;
   midinote2freq[47] = 123.47;
   midinote2freq[48] = 130.81;
   midinote2freq[49] = 138.59;
   midinote2freq[50] = 146.83;
   midinote2freq[51] = 155.56;
   midinote2freq[52] = 164.81;
   midinote2freq[53] = 174.61;
   midinote2freq[54] = 185.00;
   midinote2freq[55] = 196.00;
   midinote2freq[56] = 207.65;
   midinote2freq[57] = 220.00;
   midinote2freq[58] = 233.08;
   midinote2freq[59] = 246.94;
   midinote2freq[60] = 261.63;
   midinote2freq[61] = 277.18;
   midinote2freq[62] = 293.66;
   midinote2freq[63] = 311.13;
   midinote2freq[64] = 329.63;
   midinote2freq[65] = 349.23;
   midinote2freq[66] = 369.99;
   midinote2freq[67] = 392.00;
   midinote2freq[68] = 415.30;
   midinote2freq[69] = 440.00;
   midinote2freq[70] = 466.16;
   midinote2freq[71] = 493.88;
   midinote2freq[72] = 523.25;
   midinote2freq[73] = 554.37;
   midinote2freq[74] = 587.33;
   midinote2freq[75] = 622.25;
   midinote2freq[76] = 659.26;
   midinote2freq[77] = 698.46;
   midinote2freq[78] = 739.99;
   midinote2freq[79] = 783.99;
   midinote2freq[80] = 830.61;
   midinote2freq[81] = 880.00;
   midinote2freq[82] = 932.33;
   midinote2freq[83] = 987.77;
   midinote2freq[84] = 1046.50;
   midinote2freq[85] = 1108.73;
   midinote2freq[86] = 1174.66;
   midinote2freq[87] = 1244.51;
   midinote2freq[88] = 1318.51;
   midinote2freq[89] = 1396.91;
   midinote2freq[90] = 1479.98;
   midinote2freq[91] = 1567.98;
   midinote2freq[92] = 1661.22;
   midinote2freq[93] = 1760.00;
   midinote2freq[94] = 1864.66;
   midinote2freq[95] = 1975.53;
   midinote2freq[96] = 2093.00;
   midinote2freq[97] = 2217.46;
   midinote2freq[98] = 2349.32;
   midinote2freq[99] = 2489.02;
   midinote2freq[100] = 2637.02;
   midinote2freq[101] = 2793.83;
   midinote2freq[102] = 2959.96;
   midinote2freq[103] = 3135.96;
   midinote2freq[104] = 3322.44;
   midinote2freq[105] = 3520.00;
   midinote2freq[106] = 3729.31;
   midinote2freq[107] = 3951.07;
   midinote2freq[108] = 4186.01;
   midinote2freq[109] = 4434.92;
   midinote2freq[110] = 4698.64;
   midinote2freq[111] = 4978.03;
   midinote2freq[112] = 5274.04;
   midinote2freq[113] = 5587.65;
   midinote2freq[114] = 5919.91;
   midinote2freq[115] = 6271.93;
   midinote2freq[116] = 6644.88;
   midinote2freq[117] = 7040.00;
   midinote2freq[118] = 7458.62;
   midinote2freq[119] = 7902.13;
   midinote2freq[120] = 8372.02;
   midinote2freq[121] = 8869.84;
   midinote2freq[122] = 9397.27;
   midinote2freq[123] = 9956.06;
   midinote2freq[124] = 10548.08;
   midinote2freq[125] = 11175.30;
   midinote2freq[126] = 11839.82;
   midinote2freq[127] = 12543.85;
};

// queue storage for notes to be written to the 9919
int xs[100]; // the storage
int xw = 0;  // the write position

// add one item to the queue
void queue_one (int x) {
   xs[xw] = x;
   xw++;
}

// write the notes in the queue
void queue_write (int last) {

   // first byte is the length of the note, excluding the this length count and the duration
   printf ("   0x%02x, ", xw - 1);

   // write the queued bytes, which are the voice frequencies and volumes, and the trailing duration
   for (int i = 0; i < xw; i++) {
      printf ("0x%02x", xs[i]);
      if (!((i == (xw - 1)) && last)) {
         printf (", ");
      }
   }
   printf ("\n");

   // reset the queue
   xw = 0;
}

int queue_length () {
   return xw;
}

// value and changed flag
typedef struct {
   int num;
   int changed;
} value_t;

// type to collect midinotes that are requested to be off or on
typedef struct {
   value_t value;
   value_t off_on;
} midinote_t;

typedef struct {
   midinote_t note[NUM_VOICES];
} midinotes_t;

// storage for the midinotes
midinotes_t midinotes;

void init_midinotes () {
   for (int i = 0; i < NUM_VOICES; i++) {
      midinotes.note[i].value.num      = UNDEFINED;
      midinotes.note[i].value.changed  = FALSE;
      midinotes.note[i].off_on.num     = VOICE_OFF;
      midinotes.note[i].off_on.changed = FALSE;
   }
}

int display_comments = FALSE;

// write a 9919 note constructed from the several midi requests for on, off
void write_this_note (int t, int dur, int is_last) {

   int val;

   // print the current sync time
   if (display_comments) {
      printf ("   // sync %d\n", t);
   }

   // output only the changes
   for (int i = 0; i < NUM_VOICES; i++) {
      if (midinotes.note[i].value.changed) {
         // generate the frequency for the specified duration
         val = 111860.8 / midinote2freq[midinotes.note[i].value.num];

         // add the frequency
         queue_one (tone_freq_cmd[i] | (val & 0x0000000f));
         val = val >> 4;
         queue_one (val);
         midinotes.note[i].value.changed = FALSE;
      }

      if (midinotes.note[i].off_on.changed) {
         if (midinotes.note[i].off_on.num == VOICE_ON) {
            val = 0x00;
         } else {
            val = 0x0f;
         }
         // add the volume
         queue_one (tone_vol_cmd[i] | val);
         midinotes.note[i].off_on.changed = FALSE;
      }
   }

   // confirm there's at least one command to be written, otherwise the 9919 will not
   // be happy with a zero length command and a duration. But if that's what is
   // present, then just include volume for the first voice.
   if (queue_length () == 0) {
      if (midinotes.note[0].off_on.num == VOICE_ON) {
         val = 0x00;
      } else {
         val = 0x0f;
      }
      // add the volume
      queue_one (tone_vol_cmd[0] | val);
   }

   // add the duration
   if (!is_last) {
      queue_one ( (int) ((double) dur / (1000.0 / 60.0) * ISR_performance_factor) );
   } else {
      // last duration should be zero
      queue_one (0x00);
   }

   // output the current queue
   queue_write (is_last);
}

// capture the midinote to be turned off
void turn_off (int midinote) {
   int found = FALSE;
   for (int i = 0; i < NUM_VOICES; i++) {
      if (midinotes.note[i].value.num == midinote) {
         midinotes.note[i].off_on.num     = VOICE_OFF;
         midinotes.note[i].off_on.changed = TRUE;
         found = TRUE;
         break;
      }
   }
   assert (found);
}

// capture the midinote to be turned on
void turn_on (int midinote) {
   int found = FALSE;

   // locate any voice that still has this midinote, and if found turn on if not already on
   for (int i = 0; i < NUM_VOICES; i++) {
      if (midinotes.note[i].value.num == midinote) {
         if (midinotes.note[i].off_on.num != VOICE_ON) {
            midinotes.note[i].off_on.num     = VOICE_ON;
            midinotes.note[i].off_on.changed = TRUE;
         }
         found = TRUE;
         break;
      }
   }

   // if not still found, then locate any free voice and use it, and set it on
   if (!found) {
      for (int i = 0; i < NUM_VOICES; i++) {
         if (midinotes.note[i].off_on.num == VOICE_OFF) {
            midinotes.note[i].value.num      = midinote;
            midinotes.note[i].value.changed  = TRUE;
            midinotes.note[i].off_on.num     = VOICE_ON;;
            midinotes.note[i].off_on.changed = TRUE;
            found = TRUE;
            break;
         }
      }
   }
         
   assert (found);
}

void print_usage (const char *program) {
   printf ("%s Usage:\n", program);
   printf ("\tmidi_filename [options]\n");
   printf ("\n");
   printf ("options:\n");
   printf ("\t-b : insert buffer fill notes (0 duration notes)\n");
   printf ("\t-c : insert comments from midi translation\n");
}

// process the midi file
int main (int argc, char *argv[]) {

   char s[1024];
   char *p;
   int synctime_inwork     = -1;
   int synctime_this_line;
   char *noteon;
   char *noteoff;
   int midinote;
   int tempo = -1;
   int syncdur = -1;
   char pipe_cmd[256];

   if (argc > 1) {
      // generate the command to extract the midi info this program needs for processing
      sprintf (pipe_cmd, "mididump \"%s\" > .mididump; sort .mididump; rm .mididump", argv[1]);
   
      FILE *f = popen (pipe_cmd, "r");
      if (f) {
   
         init_midinote2freq ();
   
         init_midinotes ();
   
         printf ("const char this_sound[] = {\n");
   
         while (fgets (s, sizeof (s), f)) {
            s[strlen(s) - 1] = 0;
 
            if (display_comments) {
               printf ("// [%s]\n", s);
               fflush (stdout);
            }
 
            // handle tempo
            p = strstr (s, "Tempo =");
            if (p && (tempo == -1)) {
               p = strstr (s, "=") + 1;
               tempo = atoi (p);
               if (display_comments) {
                  printf ("   // tempo = %d\n", tempo);
               }
            }
   
            // capture whether note on or off

            // on is 9*, where * is the channel
            noteon  = strstr (s, "[9");
            if (noteon) {
               noteon += 4;
               midinote = (int) strtol (noteon, NULL, 16);
               noteon += 3;
               syncdur = (int) strtol (noteon, NULL, 16);
            }

            // off is 9*, where * is the channel
            noteoff = strstr (s, "[8");
            if (noteoff) {
               noteoff += 4;
               midinote = (int) strtol (noteoff, NULL, 16);
               noteoff += 3;
               syncdur = (int) strtol (noteoff, NULL, 16);
            }
   
            // process this note
            if (noteon || noteoff) {
   
               // capture the sync time
               synctime_this_line = atoi (s);
 
               // if sync time has changed, process it  
               if (synctime_this_line != synctime_inwork) {

/*
                  // if a larger time period has occured, process it to completion
                  while (synctime_this_line > (synctime_inwork + syncdur)) {
                     if (display_comments) {
                        printf ("// multiple sync counts skipped %d %d!\n", synctime_inwork, syncdur);
                     }
                     synctime_inwork += syncdur;
                     write_this_note (-1, tempo, FALSE);
                  }
 */    
                  // finally process this new note
                  if (synctime_inwork != -1) write_this_note (synctime_inwork, tempo, FALSE);
                  synctime_inwork = synctime_this_line;
               }
 
               // if note has been turned off, process it  
               if (noteoff) {
                  turn_off (midinote);
               }

               // if note has been turned on, process it
               if (noteon) {
                  turn_on (midinote);
               }
            }
         }

         // write the last note since there wasn't a time transition between the last set of notes and
         // the end of file
         write_this_note (-1, tempo, TRUE);

         // finally turn off all voices
         // add the volume at max
//       for (int i = 0; i < NUM_VOICES; i++) {
//          queue_one (tone_vol_cmd[i] | 0x0f);
//       }
//       queue_write (-1);

//       queue_one (0x00);
//       queue_write (1);
   
         printf ("};\n");
        
         fclose (f);
      } else {
         printf ("can't open %s\n", argv[1]);
      }
   } else {
      print_usage (argv[0]);
   }

   return 0;
}
