/* vi.c
   This program provides a simple rendition of the screen-oriented (visual) display editor.
   It mimics vi's basic capabilities, including:

   memory management
   - vi has some serious memory management, and in fact most of the memory is devoted precisely to that.

   - The primary regions are:
     - BSS
       - Shared memory control data - part of the Model (technically)
       - View data
       - Controller data
     - TEXT @ A000
       - the program itself. As of 10/20/2025, there's about 200 bytes left :-(
     - Heap @ end of TEXT segment -- note that a check is made to ensure that the heap doesn't overlap with 
       shared memory. Long term need heap and shm to work better together, or at least not allow conflicts. 
       For now this check is performed and will result in a program error message if heap and shared memory 
       overlap.
       - Document (filename, line count, and line references)
     - Shared Memory @ F000 for 4 KB
       - Mapped SAMS pages
 
   limitations
   - this program uses the deprecated dylib.* dynamic link library interface

   development: T=tested, X=developed
   - TX pick the ESC key
   - TX delete character
   - TX delete line
   - TX signal handling
   - TX word forward (w)
   - TX word backward (b)
   - TX select row number
   - TX saving existing with a filename
   - TX saving existing without a filename
   - TX saving new with a filename
   - TX saving new without a filename
   - TX quit/exit
   - TX merge lines (J)
   - TX goto last line
   - TX incorporate shmmgr
   - TX add line length protection
   - TX add document length protection (set max lines to 10)
   - TX add merge line length protection
   - TX file open failure should simply exit
   - TX bug: insert at head of line should insert at first non-whitespace
   - TX beep()
   - TX multiple operation count
   - TX multiple ops for arrows
   - TX multiple ops for delete char
   - TX multiple ops for word left/right
   - TX use console_set_pos and console_puts over file functions, in the interest of performance
   - TX mod shmmgr free and alloc / 8 to shift 3
   - TX put
   - TX yank multiple
   - TX update man page
   - TX delete multiple
   - TX delete with copy
   - TX start vi with a filename but file doesn't exist
                    
   bugs:
   - tabs will glitch the cursor position

   nice to haves:
   - read and use Unix99 binary format files for text files -- needed to use lines longer than 80 characters
   - mod shmmgr to use shift/max instead of div/mod for providing pointer to allocated space
   - add free method that will also initialize the data line to 'DELETED' so errors can be detected
   - performance improvements (current performance seems adequate...)
     - stop redrawing the INSERT text on every action
     - shmmgr - use shift and mask rather than div and mod
   - quit confirm if not saved
   - page up/down
   - overstrike editing
   - in program reading
   - while in insert mode at end of a line, such as 40 column length, the cursor is sitting on the next line. Causes no harm but would be nice if the empty row 
     appeared. Ehhh.

   change history
   09/14/2025 initial version with basic model and view
   09/18/2025 added cursor based scrolling through a document
              added reading of a file at startup given a command line argument
              reworked separation of model, view and controller
   09/18/2025 corrected multiple nit bugs
   09/29/2025 added ability to create a new line while editing
              address multiple editing edge cases
   10/01/2025 updated to use dylib to reduce program size (result was 14KB dropped to under 5 KB)
   10/02/2025 removed constant recalculation of string lengths
              added cursor draw and replacing of under cursor character
              added smart redrawing to minimize lag and flashing             
   10/03/2025 added delete line
              added a signal handler mask (ignore)
              added word forward and back
   10/04/2025 added select row number, saving, quit
   10/10/2025 added external keyboard mappings
              migrated from using the key ` as the console ESC to F1
              integrated jedimatt42 external keyboard alternatives
              added cursor movement in immediate edit mode
              corrected mishandling of LF
   10/18/2025 updated document memory management to use the shmmgr library that relies on shared memory
   10/19/2025 added line length and count protections
              finalized shmmgr incorporation for line splits and merges
   10/20/2025 added beep to flash the display
              corrected insert at head of line to insert at first non-whitespace or end of line
              lots of testing but no bugs found :-)
   10/21/2025 reduced max doc size to 4096 lines
              introduced pbuf for deleted and yanked lines
              introduced command op count
              added double key command concept, for dd, later for yy
              added operation counted cursor movement
              added operation counted word left / right movement
              added operation counted delete character
              migrated beep() to console_display_flash
   10/22/2025 added ctrl_clear_paste_buffer, ctrl_yank_line
   10/23/2025 corrected yank line loop bounds check
              added put/paste
              added TI text file line length limit. added hooks to support alternative length for Unix99 binary text files
              added backspace delete character for immediate edit mode arrow left/backspace
   10/25/2025 corrected filename handling
   10/26/2025 optimizations
   12/12/2025 reduced memory size by commenting out the memory addresses which are part of the debugging code
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdbool.h>
#include <unistd.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <console.h>
#include <dylib.h>
#include <util.h>
#include <conversion.h>
#include <math.h>

#define KEY_CR                 13                  // CR / Enter
#define KEY_ESC                27                  // Escape
#define KEY_ESC_ALT_CONSOLE_F1 3                   // f1 on a TI99 console (sadly the delete key)
#define KEY_ESC_ALT_CONSOLE_F9 15                  // f9 on console / jedimatt42 external keyboard ESC mapping
#define KEY_DOWN               10                  // key down
#define KEY_UP                 11                  // key up
#define KEY_LEFT               8                   // key left
#define KEY_RIGHT              9                   // key right

#define LINE_MAX_CHARS_U99BIN  126                 // maximum characters in a line (Unix99 binary format)  - 125 + null
#define LINE_MAX_CHARS_TI      81                  // maximum characters in a line (TI legacy text format) - 80 + null

typedef struct {                                   // definition of a line
   int len;                                        // length
   char s[LINE_MAX_CHARS_U99BIN];                  // string text
} line_t;

#define ALL_LINES_MAX        4608
#define DOC_LINES_MAX        4096                  // maximum number of lines in a document -- 3 largest shared memory segments / size of the line_t
#define PBUF_LINES_MAX        512                  // maximum number of lines in a put (paste) buffer

typedef struct {                                   // document / model definition
   char filename[FILENAME_MAX];                    // filename of the document
   int shm_ai[DOC_LINES_MAX];                      // shared memory manager allocation indices
   int count;                                      // line count in the document
   int max_line_len;                               // max line length. will either be value of LINE_MAX_CHARS_U99BIN or LINE_MAX_CHARS_TI
} doc_t;

typedef struct {                                   // put/paste buffer definition
   int shm_ai[PBUF_LINES_MAX];
   int count;
} pbuf_t;

typedef struct {                                   // vis / view definition
   int display_rows;                               // number of rows in the display
   int display_cols;                               // number of columns in the display
   int doc_line_displayed_first;                   // first line in the document being displayed
   int doc_line_displayed_last;                    // last line in the document being displayed
   int old_cursor_y;                               // the previous cursor position y, for removing the cursor
   int old_cursor_x;                               // the previous cursor position x, for removing the cursor
   char under_cursor_char[2];                      // the character under the cursor. GCC compiler issue, ensure array of 2
} vis_t;

#define EDIT_MODE_NONE    0                        // edit mode definitions
#define EDIT_MODE_INSERT  1                        // insert mode
#define EDIT_MODE_REPLACE 2                        // not currently used

#define OP_COUNT_MAX_STRLEN       6                // operation count max string length

typedef struct {                                   // controller definition
   int edit_mode;                                  // edit mode (current)
   int cursor_y;                                   // current cursor y
   int cursor_x;                                   // current cursor x
   int doc_line;                                   // current line in the document
   int doc_col;                                    // current column in the document
   bool run;                                       // run status
   char op_count_str[OP_COUNT_MAX_STRLEN];         // collected string for operation count
   int op_count_str_index;                         // index into the string
   int op_count;                                   // the op count
   int double_key_op;                              // the key representing the op that requires a double key press
} ctrl_t;

shmmgr_t doc_shmmgr;                               // shared memory manager data, part of the model/doc
doc_t *doc   = NULL;                               // model/doc
pbuf_t *pbuf = NULL;                               // put/paste buffer
vis_t vis;                                         // view
ctrl_t ctrl;                                       // controller

const char vi_cursor_char[2] = {30, 0};            // cursor definition

void view_set_cursor (int y, int x, int under_char) {    // sets the cursor

// console_set_pos (x, y);
   console_write_raw (y, x, vi_cursor_char, 1);

   vis.old_cursor_y         = y;
   vis.old_cursor_x         = x;
   vis.under_cursor_char[0] = under_char;
}

void view_unset_cursor () {                              // unsets the cursor
   if (vis.old_cursor_y >= 0) {
//    console_set_pos 
//       (vis.old_cursor_x, vis.old_cursor_y);
      console_write_raw 
         (vis.old_cursor_y, 
          vis.old_cursor_x, 
          vis.under_cursor_char, 
          1);
   }
}

void ctrl_multiple_op_init () {                       // resets the multiple operation data
   ctrl.op_count_str_index = 0;
   memset (&ctrl.op_count_str, 0x00, OP_COUNT_MAX_STRLEN);
   ctrl.op_count           = 1;
   ctrl.double_key_op      = 0;
}

void ctrl_init () {                                   // initializes the controller
   ctrl.edit_mode = EDIT_MODE_NONE;                   // set the edit mode to none
   ctrl.cursor_y  = 0;                                // a basic default position
   ctrl.cursor_x  = 0;                                // a basic default position
   ctrl.doc_line  = 0;                                // a basic default position
   ctrl.doc_col   = 0;                                // a basic default position
   ctrl.run       = true;                             // continue running
   ctrl_multiple_op_init ();
}

void view_init () {                                   // initializes the view
   struct winsize w;
   ioctl (fileno (stdout), TIOCGWINSZ, &w);           // read the display window dimensions

   vis.display_rows             = w.ws_row;           // should be set to the current screen resolution
   vis.display_cols             = w.ws_col;           // should be set to the current screen resolution
   vis.doc_line_displayed_first = 0;                  // a basic default position
   vis.doc_line_displayed_last  = DOC_LINES_MAX - 1;  // need to set to something reasonable, will be updated in the display method
   vis.old_cursor_y             = -1;                 // initialize to having the cursor_y not set
   vis.old_cursor_x             = 0;                  // initialize the cursor_x
   vis.under_cursor_char[0]     = 0;                  // initialize the under cursor character
}

void model_free () {                                  // frees the model
   shmmgr_term (&doc_shmmgr);                         // terminate shared memory usage
   if (doc) {                                         // test if the doc is allocated
      free (doc);                                     // free the model/document
      doc = NULL;                                     // initialize the model/document
   }
}

void model_alloc () {                                 // allocate space for the model

   doc = malloc (sizeof (doc_t));
   if (!doc) {
      dylib.fputs ("doc malloc failed\n", stderr);
      exit (1);
   }

   pbuf = malloc (sizeof (pbuf_t));
   if (!pbuf) {
      dylib.fputs ("pbuf malloc failed\n", stderr);
      exit (1);
   }

   unsigned int doc_addr_last =                       // determine the last address used in the A000-FFFF region
      (unsigned int) doc + sizeof (doc_t) - 1;     

   unsigned int pbuf_addr_last =
      (unsigned int) pbuf + sizeof (pbuf_t) - 1;

/*
   dylib.fputs ("doc alloc addr = ", stdout);
   dylib.fputs (uint2hex ((unsigned int) doc), stdout);
   dylib.fputs ("\n", stdout);
   dylib.fputs ("last addr = ", stdout);
   dylib.fputs (uint2hex (doc_addr_last), stdout);
   dylib.fputs ("\n", stdout);
*/

   if (doc_addr_last >= 0xf000 ||
       pbuf_addr_last >= 0xf000) {                    // ensure the model/document hasn't intruded on the shared memory page
      dylib.fputs ("program error: last doc and/or pbuf addr overlaps shared memory region\n", stderr);
      exit (1);
   }

   if (shmmgr_init                                    // initialize the shared memory manager
          (&doc_shmmgr, 
           "vi", 
           sizeof (line_t))) {          
      dylib.fputs ("shmmgr_init failed\n", stderr);
      exit (1);
   } 
}

void model_init (const char *filename) {                         // creates a new blank document

   model_alloc ();

   dylib.strcpy (doc->filename, filename);                       // retain the filename

   if (shmmgr_init (&doc_shmmgr, "vi", sizeof (line_t))) {       // initialize shared memory manager
      dylib.fputs ("shmmgr_init failed\n", stderr);
      exit (1);
   } 

   line_t *v;                                                    // pointer to an indexed value in shared memory

   doc->shm_ai[0] = shmmgr_alloc (&doc_shmmgr);                  // initialize the first line
   v      = shmmgr_map_and_return_ptr 
               (&doc_shmmgr, doc->shm_ai[0]);
   v->len = 0;        
   dylib.strcpy (v->s, ""); 

   doc->count        = 1;                                        // set the document line count to 1
   doc->max_line_len = LINE_MAX_CHARS_TI;                        // default to TI legacy text format length

   pbuf->count       = 0;                                        // set the number of lines in pbuf to zero
}

void model_read (char *filename) {                               // reads a file into the model / document
   int r = 1;
   char s[LINE_MAX_CHARS_U99BIN];

   FILE *f = dylib.fopen (filename, "r");                        // attempt the open
   if (f) {                                                      // test for success
      model_alloc ();
      doc->count = 0;                                            // set the count to zero
      line_t *v;                                                 // pointer to an indexed value in shared memory
      while (dylib.fgets (s, sizeof (s), f)) {                   // read each line in the file
         s[dylib.strcspn (s, "\r\n")] = 0x00;                    // kill off the cr/lf characters
         doc->shm_ai[doc->count] = shmmgr_alloc (&doc_shmmgr);   // allocate a line

         v = shmmgr_map_and_return_ptr                           // get the address to the line
                (&doc_shmmgr, doc->shm_ai[doc->count]); 
         dylib.strcpy (v->s, s);                                 // copy the contents into this line
         v->len = dylib.strlen (v->s);                           // retain the length of this line
         doc->count++;                                           // increment the count
      }
      dylib.fclose (f);                                          // close the file
      dylib.strcpy (doc->filename, filename);                    // retain the filename
      doc->max_line_len = LINE_MAX_CHARS_TI;                     // default to TI legacy text format length
      pbuf->count = 0;                                           // set the number of lines in pbuf to zero

      r = 0;                                                     // set success
   } else {
      model_init (filename);
   }
}

int model_write (char *filename) {                               // write the file to disk
   int r = 1;
   if (dylib.strlen (filename)) {
      FILE *f = dylib.fopen (filename, "w");
      if (f) {
         line_t *v;
         for (int r = 0; r < doc->count; r++) {
            v = shmmgr_map_and_return_ptr (&doc_shmmgr, doc->shm_ai[r]);
            dylib.fputs (v->s, f);
            dylib.fputs ("\n", f);
         }
         dylib.fclose (f);
         r = 0;
      }
   } else {
      r = 2;
   }
   return r;
}

int view_display_rows_required_direct (int len) {       // calculate the number of rows a length of characters will 
                                                        // require given the current visual settings
   int display_rows_reqd;
   
   display_rows_reqd = len / vis.display_cols;      
   if (len % vis.display_cols) {
      display_rows_reqd++;
   } 
   if (!display_rows_reqd) {
      display_rows_reqd = 1;
   } 
   return display_rows_reqd;
}

int view_display_rows_required (int doc_line) {         // calculate the display rows required for the current line
   line_t *v = shmmgr_map_and_return_ptr 
                  (&doc_shmmgr, doc->shm_ai[doc_line]);
   return view_display_rows_required_direct (v->len);
}

void view_draw (bool force_redraw) {                    // draw the view - this method is somewhat complicated and is reasonably optimized
   int screen_row = 0;
   int doc_line;
   int display_rows_reqd;

   bool full_redraw_required = false;

   // AUTO-SCROLLING

   // check for edit line before those currently displayed
   if (ctrl.doc_line < vis.doc_line_displayed_first) {
      vis.doc_line_displayed_first = ctrl.doc_line;

      full_redraw_required = true;

   // check for edit line well beyond those current displayed
   } else if (ctrl.doc_line > vis.doc_line_displayed_last + 1) {

      vis.doc_line_displayed_first = ctrl.doc_line;

      full_redraw_required = true;

   // check for edit line below those currently displayed
   } else {

      // loop until there's a first row in the doc that allows the current edit row to fully show 
      while (1) {

         // calculate the number of rows required to display the current edit row
         display_rows_reqd = 0;
         for (int r = vis.doc_line_displayed_first; r <= ctrl.doc_line; r++) {
            display_rows_reqd += view_display_rows_required (r);
         }

         // determine if it fits in the display, less the bottom row
         if (display_rows_reqd <= vis.display_rows - 1) {
            break;                                                   // it does, so exit the loop
         } else {
            vis.doc_line_displayed_first++;                          // scroll down a row in the document and try again
            full_redraw_required = true;
         }
      }
   }

   if (force_redraw) {
      full_redraw_required = true;
   }

   view_unset_cursor ();

   if (full_redraw_required) {
      console_cls ();                                                 // clear the screen
   }

   doc_line = vis.doc_line_displayed_first;                           // first line to show on first row

   line_t *v;

   while (doc_line < doc->count) {                                    // process lines until the last item in the doc is found
      display_rows_reqd = view_display_rows_required (doc_line);      // determine the number of screen rows reqd for this line
      if (screen_row + display_rows_reqd <= vis.display_rows - 1) {   // if it'll fit add it
         if (full_redraw_required || 
             (doc_line == ctrl.doc_line)) {
            console_set_pos (0, screen_row);                          // set the cursor position
            v = shmmgr_map_and_return_ptr 
                   (&doc_shmmgr, doc->shm_ai[doc_line]);
            console_puts (v->s);                                      // write the text
            if ((doc_line == ctrl.doc_line) &&
                (v->len % vis.display_cols)) {                        // write a trailing space in case of a backspace
               console_puts (" ");
            }
         }
         vis.doc_line_displayed_last = doc_line;                      // capture the last item added

         if (doc_line == ctrl.doc_line) {                             // if this is the current line, capture the cursor position
            ctrl.cursor_y = screen_row;
            ctrl.cursor_x = ctrl.doc_col;
         }
         doc_line++;                                                  // move to the next line

         screen_row += display_rows_reqd;                             // move to the next screen row that wasn't used
      } else {
         break;                                                       // line doesn't fit on the screen; stop displaying here
      }
   }

   // draw edit mode 
   // TODO: determine a way to not draw this on every refresh
   console_set_pos (0, vis.display_rows - 1);
   switch (ctrl.edit_mode) {
      case EDIT_MODE_INSERT:
         console_puts ("-- INSERT -- ");
         break;
      default:
         console_puts ("             ");
         break;
   }

   v = shmmgr_map_and_return_ptr (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);   // capture the under cursor character
   view_set_cursor (ctrl.cursor_y, ctrl.cursor_x, v->s[ctrl.doc_col]);        // set the cursor on the screen 
}

void ctrl_dealloc_line (int r) {
   shmmgr_free (&doc_shmmgr, r);
}

void ctrl_cursor_updown_final () {                                            // finalize a cursor up or down operation

   line_t *v = shmmgr_map_and_return_ptr (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);

   if (ctrl.doc_col > v->len - 1) {
      ctrl.doc_col = v->len - 1;
      if (ctrl.doc_col <= 0) {
         ctrl.doc_col = 0;
      }
   }
}

void ctrl_cursor_up (int count) {                                             // cursor up
   bool did_change = false;

   while (count) {
      if (ctrl.doc_line - 1 >= 0) {
         ctrl.doc_line--;
         ctrl_cursor_updown_final ();
         did_change = true;
      } else {
         break;
      }
      count--;
   }

   if (did_change) {
      view_draw (false);
   } else {
      console_display_flash ();
   }
}

void ctrl_cursor_down (int count) {                                           // cursor down

   bool did_change = false;

   while (count) {
      if (ctrl.doc_line + 1 < doc->count) {
         ctrl.doc_line++;
         ctrl_cursor_updown_final ();
         did_change = true;
      } else {
         break;
      }
      count--;
   }

   if (did_change) {
      view_draw (false);
   } else {
      console_display_flash ();
   }
}

void ctrl_backspace () {                                                      // perform backspace
   if (ctrl.doc_col) {
      line_t *v = shmmgr_map_and_return_ptr (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);
      int lines_reqd_before = view_display_rows_required_direct (v->len);
      char *q = v->s + ctrl.doc_col - 1;
      char *p = q + 1;
      while (*p) {
         *q = *p;
         p++;
         q++;
      }
      *q = 0x00;
      v->len--;
      ctrl.doc_col--;
      int lines_reqd_after = view_display_rows_required_direct (v->len);
      view_draw (lines_reqd_before != lines_reqd_after);
   } else {
      console_display_flash ();
   }
}

void ctrl_cursor_left (int count) {                                           // cursor left
   bool did_change = false;

   while (count) {
      if (ctrl.doc_col > 0) {
         ctrl.doc_col--;
         did_change = true;
      } else {
         break;
      }
      count--;
   }
   if (did_change) {
      view_draw (false);
   } else {
      console_display_flash ();
   }
}
   
void ctrl_cursor_right (int count) {                                          // cursor right

   bool did_change = false;

   line_t *v;

   while (count) {
      v = shmmgr_map_and_return_ptr (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);

      if (ctrl.doc_col + 1 < v->len) {
         ctrl.doc_col++;
         did_change = true;
      } else {
         break;
      }
      count--;
   }
   if (did_change) {
      view_draw (false);
   } else {
      console_display_flash ();
   }
}

void ctrl_insert_at_cursor () {                                               // start insert mode at cursor
   ctrl.edit_mode = EDIT_MODE_INSERT;

   view_draw (false);
}

void ctrl_insert_after_cursor () {                                            // start insert mode after current cursor
   ctrl.doc_col++;

   line_t *v = shmmgr_map_and_return_ptr 
                  (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);

   if (ctrl.doc_col > v->len) {
      ctrl.doc_col = v->len;
   }

   ctrl.edit_mode = EDIT_MODE_INSERT;

   view_draw (false);
}

void ctrl_insert_at_line_head () {                                            // start insert mode at head of line

   line_t *v = shmmgr_map_and_return_ptr 
                  (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);
   char *p   = skip_whitespace (v->s);

   unsigned int vs_addr  = (unsigned int) v->s;
   unsigned int ins_addr = (unsigned int) p;

   ctrl.doc_col = ins_addr - vs_addr;

   view_draw (false);
   ctrl.edit_mode = EDIT_MODE_INSERT;

   view_draw (false);
}

void ctrl_insert_row_before () {                                              // inserts a new row before the current and enters immediate edit mode

   if (doc->count < DOC_LINES_MAX) {                                          // ensure another line fits

      for (int i = doc->count; i > ctrl.doc_line; i--) {                      // move all lines at and below the cursor down by one
         doc->shm_ai[i] = doc->shm_ai[i - 1];
      }
      doc->count++;                                                           // increment the doc line count
   
      doc->shm_ai[ctrl.doc_line] = shmmgr_alloc (&doc_shmmgr);
      line_t *v = shmmgr_map_and_return_ptr 
                     (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);
      
      dylib.strcpy (v->s, "");
      v->len = 0;
   
      ctrl.doc_col = 0;                                                       // set the line column to the beginning
 
      ctrl.edit_mode = EDIT_MODE_INSERT;
   
      view_draw (true);
   } else {
      console_display_flash ();
   }
}

void ctrl_insert_row_after () {                                               // inserts a new row after the current and enters immediate edit mode

   if (doc->count < DOC_LINES_MAX) {                                          // ensure another line fits

      for (int i = doc->count; i > ctrl.doc_line; i--) {                      // move all lines at and below the cursor down by one
         doc->shm_ai[i] = doc->shm_ai[i - 1];
      }
   
      doc->count++;                                                           // increment the doc line count
      
      doc->shm_ai[ctrl.doc_line + 1] = shmmgr_alloc (&doc_shmmgr);
      line_t *v = shmmgr_map_and_return_ptr 
                     (&doc_shmmgr, doc->shm_ai[ctrl.doc_line + 1]);
      dylib.strcpy (v->s, "");
      v->len = 0;
   
      ctrl.doc_line++;  
      
      ctrl.doc_col = 0;                                                       // set the line column to the beginning

      ctrl.edit_mode = EDIT_MODE_INSERT;

      view_draw (true);
   } else {
      console_display_flash ();
   }
}

void ctrl_delete_char (int count) {                                           // deletes a count of characters from the current position

   line_t *v;

   bool did_change = false;

   bool is_complex_change = (count > 1);

   while (count) {

      v = shmmgr_map_and_return_ptr 
             (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);

      if (v->len) {
         int old_reqd_rows = view_display_rows_required (ctrl.doc_line);
   
         v = shmmgr_map_and_return_ptr 
                (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);
         char *p = v->s + ctrl.doc_col;

         char *q;

         q = p + 1;

         while (1) {
            *p = *q;
            if (!*p) {
               break;
            }
            p++;
            q++;
         }
         v->len--;
   
         int new_reqd_rows = view_display_rows_required (ctrl.doc_line);

         did_change = true;

         is_complex_change = is_complex_change || (old_reqd_rows != new_reqd_rows);

         if (ctrl.doc_col > v->len - 1) {
            if (ctrl.doc_col) {
               ctrl.doc_col--;
            }
            break;                                                            // stop deleting if the end of line has been reached
         }
      }
      count--;
   }

   if (did_change) {
      view_draw (is_complex_change);
   } else {
      console_display_flash ();
   }
}

void ctrl_clear_paste_buffer () {
   for (int i = 0; i < pbuf->count; i++) {
      shmmgr_free (&doc_shmmgr, pbuf->shm_ai[i]);
   } 
   pbuf->count = 0;
}

void ctrl_paste () {                                                          // put/paste
   if (pbuf->count &&                                                         // ensure there's lines to paste
       (doc->count + pbuf->count <= DOC_LINES_MAX)) {                         // ensure that the new doc length will not exceed the max allowed
     
      for (int i = doc->count - 1 + pbuf->count; i > ctrl.doc_line; i--) {    // move the current doc lines down, starting from the end
         doc->shm_ai[i] = doc->shm_ai[i - pbuf->count];
      }
      doc->count += pbuf->count;                                              // increment the number of lines in the doc to the new value

      line_t *v;
      line_t temp_line;
      int j = ctrl.doc_line + 1;                                              // capture the doc insert point

      for (int i = 0; i < pbuf->count; i++) {                                 // loop through pbuf
         v = shmmgr_map_and_return_ptr (&doc_shmmgr, pbuf->shm_ai[i]);        // get the first pbuf line
         dylib.memcpy (&temp_line, v, sizeof (line_t));                       // copy the contents locally
         doc->shm_ai[j] = shmmgr_alloc (&doc_shmmgr);                         // allocate space for the new line in the document
         v = shmmgr_map_and_return_ptr (&doc_shmmgr, doc->shm_ai[j]);         // get a pointer to the newly allocated space
         dylib.memcpy (v, &temp_line, sizeof (line_t));                       // copy the local contents to the newly allocated item
         j++;                                                                 // increment the position in the document
      }

      ctrl.doc_line++;                                                        // move to the top of the newly inserted rows
      ctrl.doc_col = 0;                                                       // reset the document column

      view_draw (true);                                                       // redraw the display
   } else {
      console_display_flash ();
   }
}

int ctrl_yank_line (int count) {                                              // yank lines

   int r = 0;

   ctrl_clear_paste_buffer ();

   line_t *v;
   line_t temp_line;

   int j = 0;
   if (count <= PBUF_LINES_MAX) {
      int end_line = min (ctrl.doc_line + count - 1, doc->count - 1);
      for (int i = ctrl.doc_line; i <= end_line; i++) {
         v = shmmgr_map_and_return_ptr (&doc_shmmgr, doc->shm_ai[i]);
         dylib.memcpy (&temp_line, v, sizeof (line_t));
         pbuf->shm_ai[j] = shmmgr_alloc (&doc_shmmgr);
         v = shmmgr_map_and_return_ptr (&doc_shmmgr, pbuf->shm_ai[j]);
         dylib.memcpy (v, &temp_line, sizeof (line_t));
         j++;
      }
      pbuf->count = j;
   } else {
      console_display_flash ();
      r = 1;
   }
   return r;
}

void ctrl_delete_line (int count) {                                        // deletes a line
   if (!ctrl_yank_line (count)) {
      int end_line = min (ctrl.doc_line + count - 1, doc->count - 1);      // calculate last row being deleted
      int d_count  = end_line - ctrl.doc_line + 1;                         // calculate the actual deleted lines

      int i;

      for (i = ctrl.doc_line; i <= end_line; i++) {                        // free the lines being deleted
         shmmgr_free (&doc_shmmgr, doc->shm_ai[i]);
      }

      for (i = ctrl.doc_line; i < doc->count - d_count; i++) {             // move the lines up
         doc->shm_ai[i] = doc->shm_ai[i + d_count];
      }

      doc->count -= d_count;                                               // subtract the number of deleted items from the document count

      if (!doc->count) {                                                   // handle the possibility that the document has no lines
         doc->shm_ai[0] = shmmgr_alloc (&doc_shmmgr);                      // allocate a line
         line_t *v = shmmgr_map_and_return_ptr                             // get a pointer to the line
                        (&doc_shmmgr, doc->shm_ai[0]);
         dylib.strcpy (v->s, "");                                          // initialize the line
         v->len = 0;
         doc->count = 1;                                                   // set the line count to 1
      }

      if (ctrl.doc_line > doc->count - 1) {                                // ensure the cursor is within the document
         ctrl.doc_line = doc->count - 1;
      }
   
      ctrl.doc_col = 0;
 
      view_draw (true);
   } else {
      console_display_flash ();
   }
}

void ctrl_cursor_word_left (int count) {                                      // moves the cursor left count words

   int r, c;
   char *p;

   bool did_change = false;

   while (count) {

      int sc = ctrl.doc_col;

      line_t *v;

      bool found_space     = false;
      bool found_next_word = false;
      for (r = ctrl.doc_line; r >= 0; r--) {
         v = shmmgr_map_and_return_ptr 
                (&doc_shmmgr, doc->shm_ai[r]);
         p = v->s + sc;
         for (c = sc; c >= 0; c--) {
            if (*p == ' ') {
               found_space = true;
            } else if (found_space) {
               ctrl.doc_line = r;
               ctrl.doc_col  = c;
               found_next_word = true;
               break;
            }
            p--;
         }
         if (found_next_word) {
            break;
         }
         found_space = true;                                                  // treat the end of the line as a space
         if (r) {
            v = shmmgr_map_and_return_ptr 
                   (&doc_shmmgr, doc->shm_ai[r - 1]);
            sc = v->len - 1;
            if (sc <= 0) {
               sc = 0;
            }
         }
      }
   
      if (found_next_word) {
         v = shmmgr_map_and_return_ptr 
                (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);
         p = v->s + ctrl.doc_col;
         
         for (c = ctrl.doc_col; c >= 0; c--) {
            if (*p == ' ') {
               break;
            }
            ctrl.doc_col = c;
            p--;
         }
         did_change = true;
      } else {
         break;
      }
      count--;
   }

   if (did_change) {
      view_draw (false);
   } else {
      console_display_flash ();
   }
}

void ctrl_cursor_word_right (int count) {                                     // moves the cursor right count words

   int r, c;
   char *p;

   int did_change = false;

   line_t *v;

   while (count) {
      int sc = ctrl.doc_col;

      bool found_space     = false;
      bool found_next_word = false;
      for (r = ctrl.doc_line; r < doc->count; r++) {
         v = shmmgr_map_and_return_ptr 
                (&doc_shmmgr, doc->shm_ai[r]);
         p = v->s + sc;
         for (c = sc; c < v->len; c++) {
            if (*p == ' ') {
               found_space = true;
            } else if (found_space) {
               ctrl.doc_line = r;
               ctrl.doc_col  = c;
               found_next_word = true;
               did_change = true;
               break;
            }
            p++;
         }
         if (found_next_word) {
            break;
         } 
         found_space = true;                   // treat the end of the line as a space
         sc          = 0;
      }
      if (!found_next_word) {
         break;
      }
      count--;
   }

   if (did_change) {
      view_draw (false);
   } else {
      console_display_flash ();
   }
}

void ctrl_move_to_line (char *cmd) {
   int line = atoi (cmd) - 1;                  // user indexing for lines starts at 1, internal is 0

   if (line < doc->count) {                    // ensure the request is within the document's line range
      ctrl.doc_line = line;
      ctrl.doc_col  = 0;
   } else {
      console_display_flash ();
   }
}

void ctrl_read_file (char *cmd) {              // reads a file
}

void ctrl_write_file (char *cmd) {

   char *fnp;

   fnp = dylib.strtok (cmd, " ");              // skip over the write verb
   fnp = dylib.strtok (NULL, " ");             // capture a filename, if present

   if (fnp) {                                  // if the filename was specified, capture it
      dylib.strcpy (doc->filename, fnp);
   }

   int r = model_write (doc->filename);        // write the file
   if (r) {                                    // output an error message if the file wasn't written and wait for user ack
      switch (r) {
         case 1: // cannot write file
            dylib.fputs ("Cannot write file ", stderr);
            dylib.fputs (doc->filename, stderr);
            break;
         case 2: // no filename set, can't write
            dylib.fputs ("No filename", stderr);
            break;
         default:
            break;
      }
      dylib.fputs (". ", stderr);
      char t[16];
      dylib.fgets (t, sizeof (t), stdin);
   }
}

void ctrl_exit () {
   ctrl.run = false;
}

void ctrl_merge_lines () {

   if (ctrl.doc_line + 1 < doc->count) {                                // ensure there's trailing line to be merged

      line_t *v;

      int max_len = 1;                                                  // assume the merge will require a space between the text of the two lines
      v = shmmgr_map_and_return_ptr                                     // get the first line
             (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);
      max_len += v->len;                                                // add its length
      v = shmmgr_map_and_return_ptr                                     // get the second line
             (&doc_shmmgr, doc->shm_ai[ctrl.doc_line + 1]);
      max_len += v->len;                                                // add its length

      if (max_len <= doc->max_line_len - 1) {                           // ensure the merge will fit

         // get the current line
         v = shmmgr_map_and_return_ptr (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);
         // make a copy of the current line
         line_t merged;
         dylib.memcpy (&merged, v, sizeof (line_t));
   
         char *p = merged.s + merged.len;
   
         if (merged.len) {
            char *q = p - 1;
            if (*q != ' ') {
               *p = ' ';                                                // merged lines get a space added
               p++;                                                     // move to the next position. warning: no longer null term'd
            }
         }
   
         // get the next line
         v = shmmgr_map_and_return_ptr 
                (&doc_shmmgr, doc->shm_ai[ctrl.doc_line + 1]);
   
         char *sws = skip_whitespace (v->s); 
         dylib.strcpy (p, sws);                                         // concatenate the next line at the merge point
         merged.len = dylib.strlen (merged.s);
   
         v = shmmgr_map_and_return_ptr                                  // now retrieve the original line, again
                (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);             
         dylib.memcpy (v, &merged, sizeof (line_t));                    // and copy the merged line
   
         shmmgr_free (&doc_shmmgr, doc->shm_ai[ctrl.doc_line + 1]);     // free the next line
   
         for (int r = ctrl.doc_line + 1; r < doc->count - 1; r++) {     // move up all the remaining lines in the document
            doc->shm_ai[r] = doc->shm_ai[r + 1];
         }
         doc->count--;                                                  // decrement the line count

         view_draw (true);                                              // finally redraw all
      } else {
         console_display_flash ();
      }
   } else {
      console_display_flash ();
   }
}

void ctrl_handle_colon_commands () {
 
   char cmd[80];
 
   view_unset_cursor ();                                // remove the cursor in the edit area

   console_set_pos (0, vis.display_rows - 1);  // write the colon in the immediate command area
   console_puts (":");

   // read the command

   dylib.console_gets (cmd, vis.display_cols - 2, false);

   switch (*cmd) {                                      // process the command
      case '1' ... '9':                                 // handle line numbers
         ctrl_move_to_line (cmd);
         break;
      case 'q':                                         // handle quit/exit
         ctrl_exit ();
         break;
      case 'r':                                         // handle reading a file
         ctrl_read_file (cmd);              
         break;
      case 'w':                                         // handle writing a file
         ctrl_write_file (cmd);
         break;
      default:                                          // handle the unhandle-able :-)
         break;
   }

   view_draw (true);
}

void ctrl_goto_last_line () {                           // moves to the last line of the file
   if (ctrl.doc_line != doc->count - 1) {
      ctrl.doc_line = doc->count - 1;
      ctrl.doc_col  = 0;
      view_draw (false);
   }
}

void ctrl_handle_non_edit_mode_key (int k) {                   // handle a non-edit mode key
                                                               // Notes:
                                                               // Some commands support performing an operation multiple times, some not.
                                                               // The first section of this method captures the count.
                                                               // The second section handles operations that do not require a repeated key press.
                                                               // It also handles capturing the first key press for repeated key commands.
                                                               // The last section handles repeated key commands.
                                                               // The code allows for interesting combinations of entry, such as 8dd, d8d, 8d0d.
                                                               // That latter one is a bit of a side effect but will result in the equivalent 80dd or d80d.

   if (k >= '0' && k <= '9') {                                 // capture command operation counts
      if (ctrl.op_count_str_index < OP_COUNT_MAX_STRLEN - 1) {
         ctrl.op_count_str[ctrl.op_count_str_index++] = k;
         ctrl.op_count = atoi (ctrl.op_count_str);             // a value of 1 means to perform the operation once, not do, and then repeat this many times
         if (ctrl.op_count <= 0) {                             // ensure a positive value for the command operation count exists.
            ctrl.op_count = 1;
         }
      }
   } else {                                                    // process commands

      if (!ctrl.double_key_op) {                               // process single key commands
         switch (k) {
            case KEY_LEFT:
               ctrl_cursor_left (ctrl.op_count);
               break;
            case KEY_RIGHT:
               ctrl_cursor_right (ctrl.op_count);
               break;
            case KEY_DOWN:
               ctrl_cursor_down (ctrl.op_count);
               break;
            case KEY_UP:
               ctrl_cursor_up (ctrl.op_count);
               break;
            case ':':
               ctrl_handle_colon_commands ();
               break;
            case 'a':
               ctrl_insert_after_cursor ();
               break;
            case 'b':
               ctrl_cursor_word_left (ctrl.op_count);
               break;
            case 'G':
               ctrl_goto_last_line ();
               break;
            case 'I':
               ctrl_insert_at_line_head ();
               break;
            case 'i':
               ctrl_insert_at_cursor ();
               break;
            case 'J':
               ctrl_merge_lines ();
               break;
            case 'O':
               ctrl_insert_row_before ();
               break;
            case 'o':
               ctrl_insert_row_after ();
               break;
            case 'p':
               ctrl_paste ();
               break;
            case 'w':
               ctrl_cursor_word_right (ctrl.op_count);
               break;
            case 'x':
               ctrl_delete_char (ctrl.op_count);
               break;
            default:
               ctrl.double_key_op = k;                         // capture the key that must be repeated for execution
               return;                                         // exit the method
         }
         ctrl_multiple_op_init ();
      } else {
         if (k == ctrl.double_key_op) {                        // ensure its the same command key
            switch (k) {
               case 'd':                                       // handle deleting lines
                  ctrl_delete_line (ctrl.op_count);
                  break;
               case 'y':                                       // handle yanking lines
                  ctrl_yank_line (ctrl.op_count);
                  break;
               default:                                        // unhandled double command key
                  console_display_flash (); 
                  break;
            }
         } else {                                              // first key != second key
            console_display_flash ();
         }
         ctrl_multiple_op_init ();
      }
   }
}

void ctrl_edit_crlf () {

   if (doc->count < DOC_LINES_MAX) {
      line_t *v;

      v = shmmgr_map_and_return_ptr                              // get the current line
             (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);

      char *p = v->s + ctrl.doc_col;                             // capture the pointer to the text to be moved to the next line

      line_t temp_next;                                          // allocate a temporary next line
      dylib.strcpy (temp_next.s, p);                             // copy the contents at the pointer to the next line
      temp_next.len = dylib.strlen (temp_next.s);                // set the length
   
      *p     = 0x00;                                             // null terminate the original line at the break point
      v->len = dylib.strlen (v->s);                              // set the length
   
      for (int i = doc->count; i > ctrl.doc_line + 1; i--) {     // move all lines at and below the cursor down by one
         doc->shm_ai[i]  = doc->shm_ai[i - 1];
      }
      doc->count++;                                              // increment the doc line count
   
      int new_line = ctrl.doc_line + 1;                          // capture the index of the new line
      doc->shm_ai[new_line] = shmmgr_alloc (&doc_shmmgr);        // allocate the new line
      v = shmmgr_map_and_return_ptr                              // get a pointer to the new line
             (&doc_shmmgr, doc->shm_ai[new_line]);
      dylib.memcpy (v, &temp_next, sizeof (line_t));             // copy the temp_next data to the new line
   
      ctrl.doc_line = new_line;                                  // move to the new line
      ctrl.doc_col = 0;                                          // set the line column to the beginning
   
      view_draw (true);
   } else {
      console_display_flash ();
   }
}

void ctrl_edit_escape () {                                       // exit immediate edit mode
   ctrl.edit_mode = EDIT_MODE_NONE;
   line_t *v = shmmgr_map_and_return_ptr 
                  (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);

   if (ctrl.doc_col > v->len - 1) {
      ctrl.doc_col--;
      if (ctrl.doc_col <= 0) {
         ctrl.doc_col = 0;
      }
   }
   view_draw (false);
}

void ctrl_edit_immediate_key (int k) {                           // handle immediate edit mode key typed text
   char *p;
   char r[LINE_MAX_CHARS_U99BIN];

   if (k >= 32) {

      line_t *v = shmmgr_map_and_return_ptr 
                     (&doc_shmmgr, doc->shm_ai[ctrl.doc_line]);

      if (v->len < doc->max_line_len - 1) {                      // ensure the max line length is not exceeded

         int old_reqd_rows = view_display_rows_required_direct (v->len);

         p = v->s + ctrl.doc_col;
         dylib.strcpy (r, p);
         *p = (char) k;
         p++;
         dylib.strcpy (p, r);
         v->len++;
         ctrl.doc_col++;
   
         int new_reqd_rows = view_display_rows_required_direct (v->len);
   
         view_draw (old_reqd_rows != new_reqd_rows);
      } else {
         console_display_flash ();
      }
   }
}

void ctrl_handle_edit_mode_key (int k) {                         // handle immediate edit mode keys
   switch (k) {
      case KEY_ESC:
      case KEY_ESC_ALT_CONSOLE_F1:
      case KEY_ESC_ALT_CONSOLE_F9:
         ctrl_edit_escape ();
         break;
      case KEY_CR:
         ctrl_edit_crlf ();
         break;
      case KEY_LEFT:
         ctrl_backspace ();
         break;
      case KEY_RIGHT:
         ctrl_cursor_right (1);                     
         break;
      case KEY_DOWN:
         ctrl_cursor_down (1);
         break;
      case KEY_UP:
         ctrl_cursor_up (1);
         break;
      default:
         ctrl_edit_immediate_key (k);
         break;
   }
}

int main (int argc, char *argv[]) {                              // the main method

   if (!isatty (fileno (stdout))) {                              // ensure the program is being run in a terminal
      dylib.fputs ("vi: Warning: Output is not to a terminal\n", stderr);
      exit (1);
   }

   signal (SIGINT, SIG_IGN);                                     // disable signals

   view_init ();

   if (argc == 2) {                                              // load a specified file into the doc model
      model_read (argv[1]);
   } else {                                                      // no file specified, so open a blank document in the model
      model_init ("");
   }
  
   ctrl_init ();                                                 // initialize the controller

   view_draw (true);                                             // draw the view

   char s[2];
   while (ctrl.run) {                                            // continue running until the run flag is cleared

      s[0] = console_getc ();                                    // get a keystroke

      if (ctrl.edit_mode == EDIT_MODE_NONE) {                    // route keystroke per the current edit mode
         ctrl_handle_non_edit_mode_key (s[0]);
      } else {
         ctrl_handle_edit_mode_key (s[0]);
      }
   }

   console_cls ();                                               // clear the screen on exit

   model_free ();                                                // deallocate the model, much of which is in shared memory

   return 0;                                                     // return success
}
