/* wc.c
   This program provides word, line, and byte counts.
   Usage: wc [-w] [-c] [-l] [file [... filen]]
   Options:
   -w provide word count
   -c provide byte count
   -l provide line count
   change log:
   06/23/2023 initial version
   08/25/2023 replaced use of fprintf with dylib.fputs to conserve memory usage
   12/10/2023 removed reference to string_ext.h
              added reference to conversion.h
   02/24/2024 broad changes for dylib
   03/08/2024 updated to count with long counts
              corrected word counts calculation so that it uses the characters defined in iswspace
              added printing total counts when multiple files are specified
   01/20/2025 moved fputs to dylib
*/

#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <conversion.h>
#include <stdio.h>
#include <assert.h>
#include <dylib.h>

bool do_count_chars = false; // indicates this option is on or off
bool do_count_words = false; // indicates this option is on or off
bool do_count_lines = false; // indicates this option is on or off

long af_bc = 0;              // all files byte count
long af_wc = 0;              // all files word count
long af_lc = 0;              // all files line count

// prints a long value in the format required by wc
void print_long (long v) {
   char o[10];
   char *p = (char *) ltoa (v);
   dylib.strcpy (o, "        ");
   dylib.strcpy (&o[8 - dylib.strlen (p)], p);
   dylib.fputs (o, stdout);
}

// returns true if the character is white space
bool iswspace (int c) {
   bool r;
   switch (c) {
      case 0x20:
      case 0x0c:
      case 0x0a:
      case 0x0d:
      case 0x09:
      case 0x0b:
         r = true;
         break;
      default:
         r = false;
   }
   return r;
}

// count for a single file stream
void do_count_file (const char *filename, FILE *f) {

   char s[256];  // string for input
   long bc = 0;  // byte count total
   long wc = 0;  // word count total
   long lc = 0;  // line count total
   int lwc;      // single line word count
   int i;        // loop var
   bool in_word; // indicates whether within a word (non-whitespace)

   // loop through the whole file stream
   while (dylib.fgets (s, sizeof (s), f)) {

      // count characters
      if (do_count_chars) {
         bc = bc + dylib.strlen (s);
      }

      // count words - strtok will insert null chars so the char count had to be performed before this
      if (do_count_words) {

         lwc     = 0;
         in_word = false;
         for (i = 0; i < dylib.strlen (s); i++) {
            if (iswspace (s[i])) {
               if (in_word) {
                  lwc++;
                  in_word = false;
               }
            } else {
               in_word = true;
            }
         }
         if (in_word) {
            lwc++;
         }

         wc = wc + lwc;
      }

      // count lines
      lc++; // just count the lines since an if check will be slower
   }

 
   // output based on options and provided filename
   if (do_count_lines) print_long (lc);
   if (do_count_words) print_long (wc);
   if (do_count_chars) print_long (bc);
   if (filename) {
      dylib.fputs (" ", stdout);
      dylib.fputs (filename, stdout);
   }
   dylib.fputs ("\n", stdout);

   af_bc += bc;
   af_wc += wc;
   af_lc += lc;
}

// prints usage info
void print_usage () {
   dylib.fputs ("usage: wc [-c] [-w] [-l] [file] [...file]\n", stderr);
}

int main (int argc, char *argv[]) {

   // capture the options
   int opt;
   bool error = false;

   while ((opt = getopt (argc, argv, "cwl")) != -1) {  // loop through all options
      switch (opt) {                                   // switch on an option
         case 'c':                                     // turn on counting characters
            do_count_chars = true;
            break;
         case 'w':                                     // turn on counting words
            do_count_words = true;
            break;
         case 'l':
            do_count_lines = true;                     // turn on counting lines
            break; 
         case '?':
         default:
            error = true;
            break;
      }
   }
 
   if (!error) {
      // if no options were specified, then turn them all on
      if (!(do_count_chars || do_count_words || do_count_lines)) {
         do_count_chars = true;
         do_count_words = true;
         do_count_lines = true; 
      }
     
      // adjust argc and argv removing the processed options
      argc -= optind;
      argv += optind;
   
      // process files on the command line if present or just use stdin
      if (argc) {
         // loop through all files
         for (int i = 0; i < argc; i++) {
            FILE *f = dylib.fopen (argv[i], "r"); // open the file
            if (f) {                              // if have an open file
               do_count_file (argv[i], f);        // process this file         
               dylib.fclose (f);                  // close the file
            } else {
               dylib.fputs ("wc: ", stderr);
               dylib.fputs (argv[i], stderr);
               dylib.fputs (": No such file or directory\n", stderr);
            }
         }

         // output totals if multiple files were specified
         if (argc >= 2) {
            if (do_count_lines) print_long (af_lc);
            if (do_count_words) print_long (af_wc);
            if (do_count_chars) print_long (af_bc);
            dylib.fputs (" total\n", stdout);
         }

      } else {
         do_count_file (NULL, stdin);      // process stdin
      }
   } else {
      print_usage ();
   }

   return 0;
}
